//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.UI.Controls.Common.js
//
/*globals _comma_separated_list_of_variables_*/
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax.js"/>
/// <reference path="~/Scripts/TFS/TFS.js" />
/// <reference path="~/Scripts/TFS/TFS.Core.js" />
/// <reference path="~/Scripts/TFS/TFS.Diag.js" />
/// <reference path="~/Scripts/TFS/TFS.UI.js" />
/// <reference path="~/Scripts/TFS/TFS.UI.Controls.js" />

TFS.module("TFS.UI.Controls.Common", ["TFS.Resources.Presentation", "TFS.UI", "TFS.UI.Controls",  "TFS.UI.Controls.Menus", "TFS.Core", "TFS.Diag", "TFS.Core.Ajax", "TFS.Host", "TFS.Core.Utils", "TFS.UI.Controls.Data"], function () {
    var log = TFS.Diag.log,
        Diag = TFS.Diag,
        Core = TFS.Core,
        assert = TFS.Diag.assert,
        verbose = TFS.Diag.LogVerbosity.Verbose,
        delegate = TFS.Core.delegate,
        getErrorMessage = TFS.getErrorMessage,
        Resources = TFS.Resources.Presentation,
        notificationService = TFS.Host.notificationService,
        keyCode = TFS.UI.KeyCode,
        Ajax = TFS.Core.Ajax,
        Utils = TFS.Core.Utils,
        domElem = TFS.UI.domElem,
        Controls = TFS.UI.Controls,
        MenuControls = TFS.UI.Controls.Menus;

    function extendWithout(options, toDelete) {
        var result = $.extend({}, options);

        if (toDelete) {
            $.each(toDelete, function (i, v) {
                delete result[v];
            });
        }

        return result;
    }

    function BaseComboBehavior(combo, options) {
        this.combo = combo;
        this._options = options || {};
        this._onForceHideDropPopupDelegate = delegate(this, this.onForceHideDropPopup);
    }

    BaseComboBehavior.prototype = function () {
        return {
            combo: null,
            _options: null,
            _dropPopup: null,
            _dataSource: null,
            _onForceHideDropPopupDelegate: null,
            initialize: function () {
                if (this._options.source) {
                    this.setSource(this._options.source);
                }
            },
            dispose: function () {
                this.hideDropPopup();
            },
            isDropVisible: function () {
                /*jslint eqeq: true */
                return this._dropPopup != null; //do not use triple equality here
            },
            setMode: function (value) {
                this._options.mode = value;
            },
            canType: function () {
                return true;
            },
            getDataSource: function () {
                return this._dataSource;
            },
            _getDropOptions: function () {
                return $.extend({
                    combo: this.combo,
                    anchor: this.combo.getElement(),
                    host: this.combo.getElement(),
                    dropElementAlign: "left-top",
                    dropBaseAlign: "left-bottom",
                    dataSource: this._dataSource
                }, this._options.dropOptions);
            },
            getDropWidth: function () {
                return this.combo.getElement().width();
            },
            showDropPopup: function () {
                var dropControlType = this._options.dropControlType, dropOptions;

                if (this._dropPopup) {
                    this.hideDropPopup();
                }

                if (dropControlType) {
                    dropOptions = this._getDropOptions();

                    // Width should be set late because at this point the data source is ready and
                    // some of the behaviors need to find max length item in the data source in order
                    // to find the min drop width
                    dropOptions.width = this.getDropWidth();

                    this._attachGlobalEvents();

                    this._dropPopup = dropControlType.createIn(dropOptions.host || $(document.body), dropOptions);

                    // Fires the dropShow option in the combo control whenever the dropdown shows
                    if (this._options.dropShow) {
                        this._options.dropShow(this._dropPopup);
                    }

                    return true;
                }

                return false;
            },
            hideDropPopup: function () {
                if (this._dropPopup) {
                    this._detachGlobalEvents();

                    // Fires the dropHide option in the combo control whenever the dropdown hides
                    // Dispose if the option doesn't exist, or if the function returns true
                    if (!this._options.dropHide || this._options.dropHide(this._dropPopup)) {
                        this._dropPopup.dispose();
                        this._dropPopup = null;
                        return true;
                    }
                    // Even if we have disposed of it, we still need to set it to null
                    this._dropPopup = null;
                }
                return false;
            },
            toggleDropDown: function () {
                if (this._dropPopup) {
                    this.hideDropPopup();
                }
                else {
                    this.showDropPopup();
                }
            },
            setSource: function (source) {
                if (this._dataSource) {
                    this._dataSource.setSource(source);
                }
            },
            getSelectedIndex: function () {
                return -1;
            },
            setSelectedIndex: function (selectedIndex) {
            },
            getText: function (value) {
                return this.combo.getInputText(value);
            },
            setText: function (value, fireEvent) {
                if (typeof value !== "undefined") {
                    this.combo.setInputText(value, fireEvent);
                }
            },
            upKey: function (e) {
            },
            downKey: function (e) {
                if (e.altKey) {
                    if (this.showDropPopup()) {
                        return false;
                    }
                }
            },
            pageUpKey: function (e) {
            },
            pageDownKey: function (e) {
            },
            leftKey: function (e) {
            },
            rightKey: function (e) {
            },
            keyDown: function (e) {
            },
            keyPress: function (e) {
            },
            keyUp: function (e) {
            },
            _attachGlobalEvents: function () {
                this.combo._bind(this.combo.getElement().parents(), "scroll", this._onForceHideDropPopupDelegate);
                notificationService.attachEvent("dialog-move", this._onForceHideDropPopupDelegate);
            },
            _detachGlobalEvents: function () {
                this.combo._unbind(this.combo.getElement().parents(), "scroll", this._onForceHideDropPopupDelegate);
                notificationService.detachEvent("dialog-move", this._onForceHideDropPopupDelegate);
            },
            onForceHideDropPopup: function (e) {
                this.hideDropPopup();
            }
        };
    }();

    function BaseComboDropPopup(options) {
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "combo-drop-popup"
        }, options));

        this.combo = this._options.combo;
    }

    BaseComboDropPopup.inherit(Controls.BaseControl, function () {
        return {
            combo: null,
            initialize: function () {
                this._bind("mousedown", delegate(this, this._onMouseDown));
                this.setPosition();

                this._base();
            },
            _onMouseDown: function (e) {
                this.combo.blockBlur();
            },
            setPosition: function () {
                TFS.UI.position(this._element, this._options.anchor, {
                    elementAlign: this._options.dropElementAlign,
                    baseAlign: this._options.dropBaseAlign
                });
            }
        };
    }());


    function Combo(options) {
        this.baseConstructor.call(this, $.extend({
            allowEdit: true,
            type: "list",
            treeLevel: 1,
            mode: "drop", // text, drop, delayed-drop
            dropWidth: "dynamic", // dynamic, fixed
            dropCount: 8,
            invalidCss: "invalid",
            disabledCss: "disabled",
            dropButtonHoverCss: "hover",
            inputCss: undefined,
            coreCssClass: "combo"
        }, options));
    }

    Combo.extend({
        _typeName: "tfs.combo"
    });

    Combo.extend(function () {
        var behaviors = {};
        return {
            registerBehavior: function (behaviorMode, behaviorType) {
                behaviors[behaviorMode] = behaviorType;
            },
            attachBehavior: function (combo, options) {
                var behaviorType = behaviors[combo._options.type];

                if (combo._options.type) {
                    behaviorType = behaviors[combo._options.type];
                }
                else {
                    behaviorType = BaseComboBehavior;
                }

                if (behaviorType) {
                    return new behaviorType(combo, options);
                }
                else {
                    return null;
                }
            }
        };
    }());

    Combo.inherit(Controls.BaseControl, {
        _dropPopup: null,
        _input: null,
        _behavior: null,
        _blockBlur: false,
        _currentText: "",

        _ensureBehavior: function () {
            if (this._behavior) {
                if ($.isFunction(this._behavior.dispose)) {
                    this._behavior.dispose();
                }

                this._behavior = null;
            }

            this._behavior = Combo.attachBehavior(this, this._options);
            this._behavior.initialize();

            if (!this._behavior) {
                throw Error.create(String.format("Unsupported combo behavior '{0}'.", this._options.type));
            }
        },

        _dispose: function () {
            if (this._behavior) {
                if ($.isFunction(this._behavior.dispose)) {
                    this._behavior.dispose();
                }

                this._behavior = null;
            }

            this._base();
        },

        _createIn: function (container) {
            this._base(container);
            this._input = $(domElem("input")).attr("type", "text");
            if (this._options.id) {
                this._input.attr("id", this._options.id + "_txt");
            }


            if (typeof this._options.value !== "undefined") {
                this._input.val(this._options.value);
            }

            this._decorate();
        },

        _enhance: function (element) {
            this._createElement();
            this._input = element;
            element.after(this._element);

            this._decorate(); //this will reparent our element
            this._currentText = this.getInputText();
        },

        _decorate: function () {
            var that = this,
                options = this._options,
                $element = this._element,
                $input = this._input, $inputWrap;

            $element.val = function () {
                if (!arguments.length) {
                    return that.getText();
                }
                else {
                    return that.setText(arguments[0], false);
                }
            };

            $input.attr("autocomplete", "off");
            this._bind("mousedown", delegate(this, this._onMouseDown));
            this._bind($input, "keydown", delegate(this, this._onInputKeyDown));
            this._bind($input, "keypress", delegate(this, this._onInputKeyPress));
            this._bind($input, "keyup", delegate(this, this._onInputKeyUp));
            this._bind($input, "focus", delegate(this, this._onInputFocus));
            this._bind($input, "blur", delegate(this, this._onInputBlur));
            this._bind($input, "click", delegate(this, this._onInputClick));

            if (!options.allowEdit) {
                $input.attr("readonly", true);
                $element.addClass("no-edit");
            }

            if (options.inputCss) {
                $input.addClass(options.inputCss);
            }

            $input.bind("change input", function () {
                return that.fireChangeIfNecessary();
            });

            if (options.noDropButton) {
                $element.addClass("no-background");
            }

            $inputWrap = $(domElem("div", "wrap"));
            $inputWrap.append($input);
            if (this._options.label && this._options.id) {
                // append a hidden label for screen readers use
                $inputWrap.append($(domElem("label", "hidden")).append(this._options.label).attr("For", $input.attr("id")));
            }
            $element.append($inputWrap);

            this.$dropButton = $(domElem("div", "drop"));
            this._bind(this.$dropButton, "mouseenter", function (e) { $(this).addClass(options.dropButtonHoverCss); });
            this._bind(this.$dropButton, "mouseleave", function (e) { $(this).removeClass(options.dropButtonHoverCss); });
            this._bind(this.$dropButton, "click", delegate(this, this._onDropButtonClick));

            $element.append(this.$dropButton);

            this._updateStyles();
        },

        _updateStyles: function () {
            var $element = this._element,
                $input = this._input,
                inputAvailable,
                dropAvailable;

            $element.addClass(this._options.type);

            // Enabling/disabling input according to the config.
            // Use the 'readonly' attribute (with disabled style) instead of the disabled attribute here
            // so that the text can be selected and due to an IE9 issue with selecting text in disabled
            // input controls (see vstspioneer Bug 803369)
            inputAvailable = this._options.enabled;
            if (inputAvailable && this._options.allowEdit) {
                $input.removeAttr("readonly");
            }
            else {
                $input.attr("readonly", true);
            }
            TFS.UI.addClass($input, this._options.disabledCss, !inputAvailable);

            // Hiding/showing dropbutton according to the mode
            dropAvailable = inputAvailable && (this._options.mode === "drop");
            $element.toggleClass("drop", dropAvailable && !this._options.noDropButton);
            $element.toggleClass("text", !dropAvailable);
        },

        initialize: function () {
            this._ensureBehavior();

            this._base();
        },

        getBehavior: function () {
            return this._behavior;
        },

        getText: function () {
            return this._behavior.getText();
        },

        setText: function (text, fireEvent) {
            return this._behavior.setText(text, fireEvent);
        },

        getInput: function () {
            /// <summary>Gets the input element of combo</summary>
            /// <returns type="DOMElement" />
            return this._input;
        },

        getInputText: function () {
            return this._input.val();
        },

        setInputText: function (text, fireEvent) {

            this._input.val(text);

            // Updating tooltip
            this._input.attr("title", text);

            if (fireEvent) {
                this.fireChangeIfNecessary(text);
            }
            else {
                this._currentText = text;
            }
        },

        getSelectedIndex: function () {
            return this._behavior.getSelectedIndex();
        },

        setSelectedIndex: function (selectedIndex, fireEvent) {
            this._behavior.setSelectedIndex(selectedIndex, fireEvent);
        },

        fireChangeIfNecessary: function (newValue) {
            if (typeof newValue === "undefined") {
                newValue = this.getText();
            }

            if (this._currentText !== newValue) {
                // Setting new text
                this._currentText = newValue;

                // Updating tooltip
                this._input.attr("title", newValue);

                return this._fireChange();
            }
        },

        toggleDropDown: function () {
            this._behavior.toggleDropDown();

            // Setting the focus to the input to accept the keys
            this._input.focus();

            this._fire('dropDownToggled', {
                isDropVisible: this._behavior.isDropVisible(),
                target: this._element
            });
        },

        _onDropButtonClick: function (e) {
            this.toggleDropDown();
            return false;
        },

        showDropPopup: function (e) {
            this._behavior.showDropPopup();

            // Setting the focus to the input to accept the keys
            this._input.focus();
        },

        hideDropPopup: function () {
            return this._behavior.hideDropPopup();
        },

        _onInputClick: function (e) {
            if (!this._options.allowEdit && this._options.enabled) {
                this.toggleDropDown();
            }

            return false;
        },

        _onInputFocus: function (e) {
            if (!this._blockBlur) {
                this._input.select();
            }
        },

        _onInputBlur: function (e) {
            if (!this._disposed) {
                this.fireChangeIfNecessary();
                if (this._blockBlur) {
                    this._input.focus(10);
                    this._blockBlur = false;
                }
                else {
                    this.hideDropPopup();
                }
            }
        },

        blockBlur: function () {
            this._blockBlur = true;
            this.delayExecute("blockBlur", 200, true, function () {
                this.cancelBlockBlur();
            });
        },

        cancelBlockBlur: function () {
            this._blockBlur = false;
            this.cancelDelayedFunction("blockBlur");
        },

        _onMouseDown: function (e) {
            this.blockBlur();
        },

        _onInputKeyDown: function (e) {
            if (this._behavior.keyDown(e) === false) {
                return false;
            }

            switch (e.keyCode) {
                case keyCode.PAGE_UP:
                    return this._behavior.pageUpKey(e);
                case keyCode.UP:
                    return this._behavior.upKey(e);
                case keyCode.PAGE_DOWN:
                    return this._behavior.pageDownKey(e);
                case keyCode.DOWN:
                    return this._behavior.downKey(e);
                case keyCode.LEFT:
                    return this._behavior.leftKey(e);
                case keyCode.RIGHT:
                    return this._behavior.rightKey(e);
                case keyCode.TAB:
                    this.cancelBlockBlur();
                    break;
                case keyCode.ENTER:
                case keyCode.ESCAPE:
                    this.fireChangeIfNecessary();
                    if (this.hideDropPopup()) {
                        return false;
                    }
                    break;
                default:
                    if (!this._behavior.canType()) {
                        return false;
                    }
                    break;
            }
        },

        _onInputKeyPress: function (e) {
            return this._behavior.keyPress(e);
        },

        _onInputKeyUp: function (e) {
            var result = this._behavior.keyUp(e);
            this.fireChangeIfNecessary();

            return result;
        },

        setTextSelection: function (selectionStart) {
            var range, input = this._input[0];

            if (input.setSelectionRange) {
                input.setSelectionRange(selectionStart, input.value.length);
            }
            else if (input.createTextRange) {
                range = input.createTextRange();
                range.moveStart("character", selectionStart);
                range.select();
            }
        },

        setSource: function (source) {
            this._behavior.setSource(source);
        },

        getEnabled: function () {
            return this._options.enabled === true;
        },

        setEnabled: function (value) {
            this._options.enabled = value === true;
            this._updateStyles();
        },

        getMode: function () {
            return this._options.mode;
        },

        setMode: function (value) {
            this._options.mode = value;
            this._behavior.setMode(value);
            this._updateStyles();
        },

        setType: function (type) {
            if (this._options.type !== type) {
                this.hideDropPopup();

                this._element.removeClass(this._options.type);
                this._options.type = type;
                this._ensureBehavior();
                this._updateStyles();
            }
        },

        setInvalid: function (value) {
            this._element.toggleClass(this._options.invalidCss, value);
        }
    });


    function ComboListDropPopup(options) {
        this.baseConstructor.call(this, $.extend({
            itemCss: ""
        }, options));
    }

    ComboListDropPopup.inherit(BaseComboDropPopup, function () {
        return {
            virtualizingListView: null,
            dataSource: null,
            initialize: function () {
                this.dataSource = this._options.dataSource;
                this.virtualizingListView = Controls.VirtualizingListView.enhance(this._element, $.extend(extendWithout(this._options, ["coreCssClass"]), {
                    maxRowCount: this._options.dropCount
                }));

                this._base();
            },
            selectPrev: function (page) {
                return this.virtualizingListView.selectPrev(page);
            },
            selectNext: function (page) {
                return this.virtualizingListView.selectNext(page);
            },
            getSelectedIndex: function () {
                return this.virtualizingListView.getSelectedIndex();
            },
            getSelectedValue: function () {
                return this.dataSource.getItemText(this.virtualizingListView.getSelectedIndex());
            },
            setSelectedValue: function (value) {
                var selectedIndex = value ? this.dataSource.getItemIndex(value) : -1;
                this.virtualizingListView.setSelectedIndex(selectedIndex);
            }
        };
    }());

    function ComboListBehavior(combo, options) {
        this.baseConstructor.call(this, combo, $.extend({
            dropControlType: ComboListDropPopup

        }, options));
    }

    ComboListBehavior.inherit(BaseComboBehavior, function () {
        var keyCodes = [keyCode.BACKSPACE, keyCode.TAB, keyCode.ENTER, keyCode.ESCAPE, keyCode.UP,
            keyCode.LEFT, keyCode.RIGHT, keyCode.DOWN, keyCode.HOME, keyCode.END, keyCode.DELETE];

        return {
            _enableAutoFill: false,
            _maxItemLength: null,
            _createDataSource: function () {
                return new Controls.ListDataSource(this._options);
            },
            initialize: function () {
                this._dataSource = this._createDataSource();
                this._base();
            },
            setSource: function (source) {
                this._base(source);
                this._maxItemLength = null;
            },
            _getDropOptions: function () {
                return $.extend(this._base(), {
                    selectedIndex: this._dataSource.getItemIndex(this.combo.getText()),
                    selectionChange: delegate(this, this._dropSelectionChanged)
                });
            },

            getMaxItemLength: function () {
                /// <summary>Finds the max item length inside the data source</summary>
                /// 
                var i,
                    text,
                    itemCount,
                    maxItemLength = 0,
                    dataSource = this._dataSource;

                if (this._maxItemLength === null) {
                    itemCount = dataSource.getCount(true);
                    for (i = 0; i < itemCount; i++) {
                        text = dataSource.getItemText(i, true, true);
                        maxItemLength = Math.max(text.length, maxItemLength);
                    }
                    this._maxItemLength = maxItemLength;
                }

                return this._maxItemLength;
            },

            getDropWidth: function () {
                /// <summary>Gets the drop width of this behavior</summary>

                var defaultWidth, minWidth;

                defaultWidth = this._base();

                // Calculating the suggested width
                minWidth = this.getMaxItemLength() * TFS.UI.Measurement.getUnitEx();

                // Increasing the width by 20% just in case
                minWidth *= 1.2;

                return Math.max(defaultWidth, minWidth);
            },

            _dropSelectionChanged: function (selectedIndex, accept) {
                var selectedValue = this._dropPopup.getSelectedValue();

                this.setText(selectedValue, true);

                if (accept) {
                    this.hideDropPopup();
                }
            },
            getSelectedIndex: function (value) {
                return this._dataSource.getItemIndex(value || this.getText(), false);
            },
            setSelectedIndex: function (selectedIndex, fireEvent) {
                this._setSelectedIndex(selectedIndex, fireEvent);
            },
            _setSelectedIndex: function (selectedIndex, fireEvent) {
                /// <summary>Set selected index</summary>
                /// <param name="selectedIndex" type="number">new selected index</param>
                /// <param name="fireEvent" type="boolean">flag to whether to fire index changed</param>
                this.setText(this._dataSource.getItemText(selectedIndex, true), fireEvent);
            },
            setText: function (value, fireEvent) {
                var changed;

                if (fireEvent) {
                    changed = value != this.getText();
                }

                this._base(value, fireEvent);

                if (fireEvent && changed) {
                    if (this._options.indexChanged) {
                        this._options.indexChanged.call(this, this.getSelectedIndex(value));
                    }
                }
            },
            upKey: function (e) {
                if (this._options.mode !== "text") {
                    if (this._base(e) == false) {
                        return false;
                    }

                    return this.selectPrev();
                }
            },
            downKey: function (e) {
                if (this._options.mode !== "text") {
                    if (this._base(e) == false) {
                        return false;
                    }

                    return this.selectNext();
                }
            },
            pageUpKey: function (e) {
                if (this._options.mode !== "text") {
                    if (this._base(e) == false) {
                        return false;
                    }

                    return this.selectPrev(true);
                }
            },
            pageDownKey: function (e) {
                if (this._options.mode !== "text") {
                    if (this._base(e) == false) {
                        return false;
                    }

                    return this.selectNext(true);
                }
            },
            keyDown: function (e) {
                if (this._options.mode !== "text") {
                    if (e.keyCode === 229) { //IME enabled
                        this._enableAutoFill = false;
                    }
                    else if (e.ctrlKey && (e.keyCode === 89 || e.keyCode === 90)) { //CTRL + Z or CTRL + Y disable undo/redo
                        return false;
                    }
                }
            },
            keyPress: function (e) {
                if (this._options.mode !== "text") {
                    // default behavior for autofill
                    if (typeof this._options.autoComplete === 'undefined') {
                        this._enableAutoFill = true;
                    }
                    else {// If there is an option specified, used that value
                        this._enableAutoFill = this._options.autoComplete;
                    }

                    return true;
                }
            },
            keyUp: function (e) {
                var key;

                if (this._options.mode !== "text") {
                    key = e.keyCode || e.charCode;
                    if (this._enableAutoFill) {
                        if (key >= 47 || key == keyCode.SPACE) {
                            this._tryAutoFill(e.target);
                        }

                        this._enableAutoFill = false;
                    }

                    return true;
                }
            },
            selectPrev: function (page) {
                var selectedIndex;

                if (this._dropPopup) {
                    if (this._dropPopup.selectPrev(page)) {
                        return false;
                    }
                }
                else if (!page) {
                    selectedIndex = this._dataSource.nextIndex(this.getSelectedIndex(), -1, true);
                    if (selectedIndex >= 0) {
                        // set selected index and fire event that it changed
                        this._setSelectedIndex(selectedIndex, true);
                        return false;
                    }
                }
            },
            selectNext: function (page) {
                var selectedIndex;

                if (this._dropPopup) {
                    if (this._dropPopup.selectNext(page)) {
                        return false;
                    }
                }
                else if (!page) {
                    selectedIndex = this._dataSource.nextIndex(this.getSelectedIndex(), 1, true);
                    if (selectedIndex >= 0) {
                        // set selected index and fire event that it changed
                        this._setSelectedIndex(selectedIndex, true);
                        return false;
                    }
                }
            },
            _tryAutoFill: function () {
                var index, currentText, itemText;

                TFS.Diag.logTracePoint("Combo._tryAutoFill.start");

                currentText = this.combo.getText();

                index = this._dataSource.getItemIndex(currentText, true, true);

                // Selecting text if the input text matches any value
                if (index >= 0) {
                    itemText = this._dataSource.getItemText(index, true);
                }

                if (itemText) {
                    this.combo.setText(itemText, true);
                    this.combo.setTextSelection(currentText.length);
                }

                if (this._dropPopup) {
                    this._dropPopup.setSelectedValue(itemText);
                }

                TFS.Diag.logTracePoint("Combo._tryAutoFill.complete");
            }
        };
    }());


    function ComboTreeDropPopup(options) {
        this.baseConstructor.call(this, $.extend({
            cssClass: "tree",
            createItem: delegate(this, this._createItem),
            itemClick: delegate(this, this._onItemClick)
        }, options));
    }

    ComboTreeDropPopup.inherit(ComboListDropPopup, function () {
        return {
            _createItem: function (index) {
                var node = this.dataSource.getItem(index), $li, $div;

                $li = $(domElem("li", this._options.nodeCss));

                if (node.hasChildren()) {
                    $li.addClass(node.expanded ? "expanded" : "collapsed");
                }

                $div = $(domElem("div", "node-content"))
                    .css("padding-left", node.level(true) * 10);

                $(domElem("span", "node-img icon"))
                    .appendTo($div);

                if (this._options.showIcons && node.icon) {
                    $(domElem("span", node.icon))
                    .appendTo($div);
                }

                $div.append($(domElem("span", "text")).text(node.text || "").attr("title", node.text || ""));
                $li.append($div);
                return $li;
            },
            _onItemClick: function (e, itemIndex, $target, $li) {
                var node;

                if ($target.hasClass("node-img")) {
                    node = this.dataSource.getItem(itemIndex);
                    if (node.hasChildren()) {
                        if (node.expanded) {
                            this.dataSource.collapseNode(node);
                        }
                        else {
                            this.dataSource.expandNode(node);
                        }

                        this.virtualizingListView.update();
                        return false;
                    }
                }
            },
            _getSelectedNode: function () {
                return this.dataSource.getItem(this.virtualizingListView.getSelectedIndex());
            },
            expandNode: function () {
                var node = this._getSelectedNode();

                if (node && node.hasChildren() && !node.expanded) {
                    this.dataSource.expandNode(node);
                    this.virtualizingListView.update();
                    return true;
                }

                return false;
            },
            collapseNode: function () {
                var node = this._getSelectedNode();

                if (node && node.hasChildren() && node.expanded) {
                    this.dataSource.collapseNode(node);
                    this.virtualizingListView.update();
                    return true;
                }

                return false;
            }
        };
    }());

    function ComboTreeBehavior(combo, options) {
        this.baseConstructor.call(this, combo, $.extend({
            dropControlType: ComboTreeDropPopup,
            sepChar: "\\",
            treeLevel: 2
        }, options));
    }

    ComboTreeBehavior.inherit(ComboListBehavior, function () {
        return {
            _createDataSource: function () {
                return new Controls.TreeDataSource(this._options);
            },
            canType: function () {
                return this._dropPopup == null; //null or undefined
            },
            leftKey: function (e) {
                if (this._dropPopup) {
                    this._dropPopup.collapseNode();

                    return false;
                }
            },
            rightKey: function (e) {
                if (this._dropPopup) {
                    this._dropPopup.expandNode();

                    return false;
                }
            },
            keyUp: function (e) {
                if (this._dropPopup) {
                    return false;
                }
                else {
                    return this._base(e);
                }
            }
        };
    }());


    // TODO: This should be in a better location, like TFS.debug.js where we override MicrosoftAjax date and number glob functions
    // or possibly in TFS.Core.Utils
    function CalendarDate(date) {
        /// <summary>Represents a date object which takes different calendar types other than
        /// GregorianCalendar in to account as well (like HijriCalendar, ThaiBuddhistCalendar)
        ///
        /// If the calendar of current culture is other than GregorianCalendar, it tries to convert
        /// the date into appropriate calendar using converter if any exists. If there is no need
        /// for conversion, it falls back to the base Date object.
        ///</summary>
        /// <param name="date" type="Date">Base date object. If nothing specified, DateTime.Now is used.</param>
        this._setDate(date || new Date());
    }

    CalendarDate.extend({
        DAY_IN_MILLISECONDS: 86400000,
        DAYS_IN_A_WEEK: 7,

        toGregorian: function (year, month, day) {
            /// <summary>Creates a date object using the specified date parts by trying to use current culture's
            /// calendar converter.</summary>
            var convert = Sys.CultureInfo.CurrentCulture.dateTimeFormat.Calendar.convert;
            if (convert) {
                return convert.toGregorian(year, month, day);
            }
            return new Date(year, month, day);
        },
        fromGregorian: function (date) {
            /// <summary>Creates a converted date using the specified gregorian date by trying to use current
            /// culture's calendar converter.</summary>
            var convert = Sys.CultureInfo.CurrentCulture.dateTimeFormat.Calendar.convert;
            if (convert) {
                return convert.fromGregorian(date);
            }
        }
    });

    CalendarDate.prototype = {

        _baseDate: null,            // Gregorian date
        _convertedDate: null,       // Converted date

        _setDate: function (date) {
            this._baseDate = date;
            this._convertedDate = CalendarDate.fromGregorian(date);
        },

        getBaseDate: function () {
            /// <summary>Gets the base date</summary>
            /// <returns type="Date" />
            return this._baseDate;
        },
        setTime: function (date) {
            /// <summary>Sets a new time using the specified date value</summary>
            /// <param name="date" type="Date">Date can be date object or ticks</param>
            date = date instanceof Date ? date : new Date(date);
            this._setDate(date);
        },
        getTime: function () {
            /// <summary>Gets the base date in ticks</summary>
            return this.getBaseDate().getTime();
        },
        getDay: function () {
            /// Gets the day of the week (from 0 to 6) of this calendar date
            return this.getBaseDate().getDay();
        },

        getFullYear: function (converted) {
            /// <summary>Gets the year of this calendar date</summary>
            /// <param name="converted" type="bool">If specified true, uses converted date. Otherwise, it uses the base date.</param>
            if (converted === true && this._convertedDate) {
                return this._convertedDate[0];
            }
            return this.getBaseDate().getFullYear();
        },
        getMonth: function (converted) {
            /// <summary>Gets the month of this calendar date</summary>
            /// <param name="converted" type="bool">If specified true, uses converted date. Otherwise, it uses the base date.</param>
            if (converted === true && this._convertedDate) {
                return this._convertedDate[1];
            }
            return this.getBaseDate().getMonth();
        },
        getDate: function (converted) {
            /// <summary>Gets the day of this calendar date in existing month</summary>
            /// <param name="converted" type="bool">If specified true, uses converted date. Otherwise, it uses the base date.</param>
            if (converted === true && this._convertedDate) {
                return this._convertedDate[2];
            }
            return this.getBaseDate().getDate();
        },

        jumpToMonthStart: function () {
            /// <summary>Jumps to the start of the month using the converted date</summary>
            var year,
                month,
                date,
                monthStart;

            year = this.getFullYear();
            month = this.getMonth();
            date = this.getDate();

            monthStart = new CalendarDate(new Date(year, month, date, 12)); //first day of the month 12 PM to address daylight saving
            this.setTime(monthStart.getTime() - CalendarDate.DAY_IN_MILLISECONDS * (monthStart.getDate(true) - 1));
            return this;
        },
        jumpToWeekStart: function () {
            /// <summary>Jumps to the start of the week using the converted date</summary>
            var dayDiff,
                firstDayOfWeek,
                weekStart;

            firstDayOfWeek = Sys.CultureInfo.CurrentCulture.dateTimeFormat.FirstDayOfWeek;
            dayDiff = (CalendarDate.DAYS_IN_A_WEEK - firstDayOfWeek + this.getDay()) % CalendarDate.DAYS_IN_A_WEEK;
            this.setTime(this.getTime() - dayDiff * CalendarDate.DAY_IN_MILLISECONDS);
            return this;
        },
        nextDay: function () {
            /// <summary>Advances current date to the next day</summary>
            this.setTime(this.getTime() + CalendarDate.DAY_IN_MILLISECONDS);
            return this;
        },
        prevMonth: function () {
            /// <summary>Advances current date to the previous month</summary>
            this._increment(0, -1);
            return this;
        },
        nextMonth: function () {
            /// <summary>Advances current date to the next month</summary>
            this._increment(0, 1);
            return this;
        },
        prevYear: function () {
            /// <summary>Advances current date to the previous year</summary>
            this._increment(-1, 0);
            return this;
        },
        nextYear: function () {
            /// <summary>Advances current date to the next year</summary>
            this._increment(1, 0);
            return this;
        },

        _increment: function (yearIncrement, monthIncrement) {
            var month = this.getMonth(true),
                year = this.getFullYear(true);

            if (monthIncrement > 0) {
                if (++month > 11) {
                    month = 0;
                    year++;
                }
            }
            else if (monthIncrement < 0) {
                if (--month < 0) {
                    month = 11;
                    year--;
                }
            }

            year += yearIncrement;

            this.setTime(CalendarDate.toGregorian(year, month, 1));
        },

        equals: function (date) {
            /// <summary>Checks whether the specified date and current date correspond to the same day</summary>
            /// <param name="date" type="Date">Date object to check</param>
            /// <returns type="Boolean" />
            return (this.getFullYear() === date.getFullYear()
                && this.getMonth() === date.getMonth()
                && this.getDate() === date.getDate());
        }
    };

    function DatePanel(options) {
        this.baseConstructor.call(this, $.extend({ coreCssClass: "date-panel" }, options));
    }

    DatePanel.inherit(Controls.BaseControl, function () {
        return {
            _date: null,
            _selectedDate: null,

            initialize: function () {
                this._selectedDate = this._options.selectedDate;
                this._date = this._selectedDate || new Date();

                this._bind("click", delegate(this, this._onClick));

                this._draw(this._date);

                this._base();
            },
            _draw: function (calendarDate) {

                var element = this._element,
                    todayFormat = this._options.todayFormat || "d",
                    $table,
                    $row,
                    $cell;

                $table = $("<table />").attr("cellSpacing", 0).attr("cellPadding", 0).attr("border", 0);
                $row = $("<tr />").addClass("quick-nav");

                $("<a />")
                    .addClass("prev-month")
                    .attr("href", "#")
                    .append($("<span/>").addClass("icon icon-drop-left"))
                    .appendTo($("<td/>")
                                .addClass("prev-month")
                                .appendTo($row));

                $cell = $("<td/>").addClass("month-year").appendTo($row);

                // Rendering date in year-month pattern ("MMMM yyyy")
                $cell.text(calendarDate.localeFormat("y", true));

                $("<a />")
                    .addClass("next-month")
                    .attr("href", "#")
                    .append($("<span/>").addClass("icon icon-drop-right"))
                    .appendTo($("<td/>")
                                .addClass("next-month")
                                .appendTo($row));

                $table.append($row);

                $row = $("<tr />");
                $cell = $("<td />").attr("colspan", 6).addClass("days");
                $cell.append(this._drawCalendarTable(calendarDate));
                $row.append($cell);

                $table.append($row);

                $row = $("<tr />");
                $cell = $("<td />").attr("colspan", 6).addClass("legend");

                // Rendering today link. Default is ShortDatePattern ("dd/MM/yy")
                $("<a />")
                    .addClass("today")
                    .attr("href", "#")
                    .append($("<span/>").addClass("today"))
                    .append(String.format(Resources.TodayTitle, (new Date()).localeFormat(todayFormat)))
                    .appendTo($cell);

                $row.append($cell);
                $table.append($row);

                element.empty().append($table);
            },

            _drawCalendarTable: function (date) {
                var calendarDate,
                    month,
                    $calendarTable,
                    $row,
                    $cell,
                    i, j,
                    dtf = Sys.CultureInfo.CurrentCulture.dateTimeFormat;

                calendarDate = new CalendarDate(date);

                // Keeping the month value to check later on to see whether we are in the same month or not
                month = calendarDate.getMonth(true);

                // Jumping to month start
                calendarDate.jumpToMonthStart();

                // Jumping to week start
                calendarDate.jumpToWeekStart();

                $calendarTable = $("<table />").attr("cellSpacing", 0).attr("border", 0);
                $row = $("<tr />").addClass("day-names");

                for (i = 0; i < 7; i++) {
                    $cell = $("<td />");
                    $cell.text(dtf.ShortestDayNames[(dtf.FirstDayOfWeek + i) % CalendarDate.DAYS_IN_A_WEEK]);
                    $row.append($cell);
                }

                $calendarTable.append($row);

                for (i = 0; i < 6; i++) {
                    $row = $("<tr />").addClass("days");

                    for (j = 0; j < CalendarDate.DAYS_IN_A_WEEK; j++) {
                        $cell = $("<td />").addClass("date-cell");

                        $("<a />")
                            .addClass("date")
                            .attr("href", "#")
                            .data("date", calendarDate.getTime())
                            .text(calendarDate.getDate(true))
                            .appendTo($cell);

                        if (calendarDate.getMonth(true) !== month) {
                            $cell.addClass("other-month");
                        }

                        if (calendarDate.equals(new Date())) { // date equals today?
                            $cell.addClass("today");
                        }

                        if (this._selectedDate instanceof Date) { // date equals selected date?
                            if (calendarDate.equals(this._selectedDate)) {
                                $cell.addClass("selected");
                            }
                        }
                        else if (calendarDate.equals(date)) { // date equals month start
                            $cell.addClass("selected");
                        }

                        $row.append($cell);

                        // Advance to next day
                        calendarDate.nextDay();
                    }

                    $calendarTable.append($row);
                }

                return $calendarTable;
            },

            _onClick: function (e) {
                var $target, $link, $dateCell;

                $target = $(e.target);
                $link = $target.closest("a");

                if ($link.length) {
                    if ($link.hasClass("prev-month")) {
                        this.prevMonth();
                    }
                    else if ($link.hasClass("next-month")) {
                        this.nextMonth();
                    }
                    else if ($link.hasClass("prev-year")) {
                        this.prevYear();
                    }
                    else if ($link.hasClass("next-year")) {
                        this.nextYear();
                    }
                    else if ($link.hasClass("today")) {
                        this.selectDate(new Date());
                    }
                    else if ($link.hasClass("date")) {
                        this.selectDate(new Date($link.data("date")));
                    }

                    return false;
                }
                else {
                    $dateCell = $target.closest(".date-cell");
                    if ($dateCell.length) {
                        this.selectDate(new Date($dateCell.find("a").data("date")));

                        return false;
                    }
                }
            },
            prevMonth: function () {
                var calendarDate = new CalendarDate(this._date);
                this._date = calendarDate.prevMonth().getBaseDate();
                this._draw(this._date);
            },
            nextMonth: function () {
                var calendarDate = new CalendarDate(this._date);
                this._date = calendarDate.nextMonth().getBaseDate();
                this._draw(this._date);
            },
            prevYear: function () {
                var calendarDate = new CalendarDate(this._date);
                this._date = calendarDate.prevYear().getBaseDate();
                this._draw(this._date);
            },
            nextYear: function () {
                var calendarDate = new CalendarDate(this._date);
                this._date = calendarDate.nextYear().getBaseDate();
                this._draw(this._date);
            },
            selectDate: function (date) {
                this.setSelectedDate(date);
                if (this._options.onSelectDate) {
                    this._options.onSelectDate();
                }
            },
            setSelectedDate: function (date) {
                this._selectedDate = this._date = date;
                this._draw(this._selectedDate);
                this._fireChange();
            },
            getSelectedDate: function () {
                return this._selectedDate;
            }
        };
    }());

    function ComboDateDropPopup(options) {
        this.baseConstructor.call(this, $.extend({}, options));
    }

    ComboDateDropPopup.inherit(BaseComboDropPopup, function () {
        return {
            _datePanel: null,
            _selectedDate: null,
            initialize: function () {
                this._selectedDate = this._options.selectedDate;

                this._datePanel = DatePanel.createIn(this._element, $.extend(extendWithout(this._options, ["coreCssClass"]), {
                    change: delegate(this, this._onChange)
                }))

                this._base();
            },
            getSelectedDate: function () {
                return this._selectedDate;
            },
            setSelectedDate: function (date) {
                this._selectedDate = date;
                if (this._datePanel) {
                    this._datePanel.setSelectedDate(date);
                }
            },
            _onChange: function (e) {
                this._selectedDate = this._datePanel.getSelectedDate();
                this._fireChange();
                return false;
            }
        };
    }());

    function ComboDateBehavior(combo, options) {
        this.baseConstructor.call(this, combo, $.extend({
            dropControlType: ComboDateDropPopup,
            dateTimeFormat: Sys.CultureInfo.CurrentCulture.dateTimeFormat.ShortDatePattern
        }, options));
    }

    ComboDateBehavior.inherit(BaseComboBehavior, function (undefined) {
        return {
            initialize: function () {
                this._base();
            },
            canType: function () {
                return this._dropPopup == null; //null or undefined
            },
            _getDropOptions: function () {
                var selectedDate, text, that = this;

                text = $.trim(this.combo.getText());
                if (text) {
                    selectedDate = Utils.dateUtils.parseDateString(text, this._options.parseFormat, true);
                }
                else if (!selectedDate && this._options.getInitialDate) {
                    selectedDate = this._options.getInitialDate(this.combo);
                }

                return $.extend(this._base(), {
                    dropElementAlign: "right-top",
                    dropBaseAlign: "right-bottom",
                    selectedDate: selectedDate,
                    onSelectDate: function () {
                        that.hideDropPopup();
                    },
                    change: delegate(this, this._onChange)
                });
            },
            getDropWidth: function () {
                return undefined;
            },
            _onChange: function () {
                var selectedDate = this._dropPopup.getSelectedDate();

                if (selectedDate) {
                    // we need locale fromat here because format use culture invariant which cause the date displayed to be UTC
                    this.setText(selectedDate.localeFormat(this._options.dateTimeFormat, true), true);
                }
                return false;
            },
            _getSelectedDate: function () {
                return this._dropPopup.getSelectedDate() || new Date();
            },
            _addDays: function (date, days) {
                var newDate = new Date();
                newDate.setTime(date.getTime() + days * 86400000);
                return newDate;
            },
            _getMonthLength: function (month, year) {
                var date;
                month = (month + 12) % 12 + 1; // switch to one based to make the function more readable. add 12 to fix any negative numbers
                switch (month) {
                    case 4:
                    case 6:
                    case 9:
                    case 11:
                        return 30;
                    case 2:
                        {
                            if (year % 4 === 0) {
                                date = new Date(year, 1, 29);
                                // check if this date has 1 as the month then it is really 29 days
                                if (date.getMonth() === 1) {
                                    return 29;
                                }
                            }

                            return 28;
                        }
                    default:
                        return 31;

                }
            },
            upKey: function (e) {
                if (this._dropPopup) {
                    this._dropPopup.setSelectedDate(this._addDays(this._getSelectedDate(), -7));
                    return false;
                }
            },
            downKey: function (e) {
                if (this._base(e) === false) {
                    return false;
                }
                if (this._dropPopup) {
                    this._dropPopup.setSelectedDate(this._addDays(this._getSelectedDate(), 7));
                    return false;
                }
            },
            pageUpKey: function (e) {
                var date;
                if (this._dropPopup) {
                    date = this._getSelectedDate();
                    this._dropPopup.setSelectedDate(this._addDays(date, -1 * this._getMonthLength(date.getMonth() - 1, date.getFullYear())));
                    return false;
                }
            },
            pageDownKey: function (e) {
                var date;
                if (this._dropPopup) {
                    date = this._getSelectedDate();
                    this._dropPopup.setSelectedDate(this._addDays(date, this._getMonthLength(date.getMonth(), date.getFullYear())));
                    return false;
                }
            },
            leftKey: function (e) {
                if (this._dropPopup) {
                    this._dropPopup.setSelectedDate(this._addDays(this._getSelectedDate(), -1));
                    return false;
                }
            },
            rightKey: function (e) {
                if (this._dropPopup) {
                    this._dropPopup.setSelectedDate(this._addDays(this._getSelectedDate(), 1));
                    return false;
                }
            }
        };
    }());

    function DatePicker(options) {
        this.baseConstructor.call(this, $.extend({
            type: "date-time"
        }, options));
    }


    function ComboMultiValueDropPopup(options) {
        this.baseConstructor.call(this, $.extend({
            cssClass: "multi-value",
            createItem: delegate(this, this._createItem),
            itemClick: delegate(this, this._onItemClick)
        }, options));

        this._checkStates = {};

        if (!this._options.id) {
            this._options.id = "cmvdp_" + Controls.getId();
        }
    }

    ComboMultiValueDropPopup.inherit(ComboListDropPopup, function () {
        return {
            _checkStates: null,
            initialize: function () {
                var parts = this.combo.getText().split(this._options.sepChar), self = this;

                $.each(parts, function (i, v) {
                    v = $.trim(v);
                    if (v) {
                        self._checkStates[v] = true;
                    }
                });

                this._base();
            },
            _createItem: function (index) {
                var itemText = this.dataSource.getItemText(index), $li, $cb, id = this._options.id + "_cb" + index;

                $li = $(domElem("li", this._options.nodeCss));

                $cb = $("<input />")
                            .data("value", itemText)
                            .attr("type", "checkbox")
                            .attr("id", id)
                            .data("index", index)
                            .appendTo($li);

                if (this._checkStates[itemText]) {
                    $cb.attr("checked", true);
                }

                $li.append($(domElem("label")).attr("for", id).text(itemText || ""));

                return $li;
            },
            getValue: function () {
                var self = this;
                return $.map(this.dataSource.getItems(), function (itemText) {
                    if (self._checkStates[itemText]) {
                        return itemText;
                    }
                }).join(this._options.joinChar || (this._options.sepChar + " "));
            },
            _onItemClick: function (e, itemIndex, $target, $li) {
                var oldValue, newValue,
                itemText = this.dataSource.getItemText(itemIndex);

                oldValue = Boolean(this._checkStates[itemText]);
                newValue = Boolean($li.find("input").attr("checked"));
                this._checkStates[itemText] = newValue;

                if (oldValue != newValue) {
                    this._fireChange();
                }
            },
            toggleCheckbox: function (selectedIndex) {
                var id = this._options.id + "_cb" + selectedIndex, $cb, itemText;

                itemText = this.dataSource.getItemText(selectedIndex);
                $cb = $("#" + id);

                if ($cb.attr("checked")) {
                    $cb.attr("checked", false);
                    this._checkStates[itemText] = false;
                }
                else {
                    $cb.attr("checked", true);
                    this._checkStates[itemText] = true;
                }
            }
        };
    }());

    function ComboMultiValueBehavior(combo, options) {
        this.baseConstructor.call(this, combo, $.extend({
            dropControlType: ComboMultiValueDropPopup,
            sepChar: ",",
            joinChar: ", "
        }, options));
    }

    ComboMultiValueBehavior.inherit(ComboListBehavior, function () {
        return {
            canType: function () {
                return this._dropPopup == null; //null or undefined
            },
            _getDropOptions: function () {
                return $.extend(this._base(), {
                    sepChar: this._options.sepChar,
                    joinChar: this._options.joinChar,
                    selectedIndex: -1,
                    selectionChange: false,
                    change: delegate(this, this._onChange)
                });
            },
            _onChange: function () {
                this.setText(this._dropPopup.getValue(), true);

                return false;
            },
            keyDown: function (e) {
                var selectedIndex;
                if (this._dropPopup) {
                    if (e.keyCode == keyCode.SPACE) {
                        selectedIndex = this._dropPopup.getSelectedIndex();

                        if (selectedIndex >= 0) {
                            this._dropPopup.toggleCheckbox(selectedIndex);
                            this.setText(this._dropPopup.getValue(), true);
                        }
                    }
                }
            }
        };
    }());

    function CollapsiblePanel(options) {
        this.baseConstructor.call(this, $.extend({
            "collapsed": true,
            "iconCss": "icon",
            "iconCollapseCss": "icon-tree-collapsed",
            "iconExpandCss": "icon-tree-expanded"
        }, options));
    }

    CollapsiblePanel.extend({
        _typeName: "tfs.collapsiblepanel"
    });

    CollapsiblePanel.inherit(Controls.BaseControl, {
        _dynamicContents: null,
        _header: null,
        _content: null,

        _createControl: function () {
            var options = this._options, $collapse;

            if (options.hoverCss) {
                this._element.hover(function () {
                    $(this).addClass(options.hoverCss);
                },
                function () {
                    $(this).removeClass(options.hoverCss);
                });
            }

            this._header = $(domElem("div", "tfs-collapsible-header"));
            if (options.headerCss) {
                this._header.addClass(options.headerCss);
            }

            $collapse = $(domElem("a", "tfs-collapsible-collapse"))
                .attr("href", "#")
                .addClass(options.iconCss)
                .addClass(options.collapsed ? options.iconCollapseCss : options.iconExpandCss)
                .bind("click.CollapsiblePanel", delegate(this, this.onExpandCollapseClick));

            if (options.collapseCss) {
                $collapse.addClass(options.collapseCss);
            }

            this._header.append($collapse);
            this._element.append(this._header);

            this._content = $(domElem("div", "tfs-collapsible-content"));
            if (options.contentCss) {
                this._content.addClass(options.contentCss);
            }

            if (options.collapsed) {
                this._content.hide();
            }

            this._element.append(this._content);

            return this._element;
        },

        _createIn: function (container) {
            this._base(container);

            this._createControl();
        },

        _enhance: function (element) {
            var title;

            this._options.collapsed = element.hasClass("collapsed");

            this._createElement();
            this._createControl();

            element.after(this._element);
            this.appendContent(element);

            if (typeof this._options.headerText !== "undefined") {
                if (this._options.headerText) {
                    this.appendHeaderText(this._options.headerText);
                }
            }
            else if (typeof this._options.headerContent !== "undefined") {
                if (this._options.headerContent) {
                    this.appendHeader(this._options.headerContent);
                }
            }
            else {
                title = element.attr("title");

                if (title) {
                    if (element.hasClass("header")) {
                        element.removeClass("header");
                        this._header.addClass("header");
                    }

                    this.appendHeader(title);
                }
            }
        },

        appendHeaderText: function (headerText) {
            /// <summary>Appends the specified plain text to the header section of the CollapsiblePanel</summary>
            /// <param name="header" type="Selector">Content to append to the header section</param>
            /// <returns type="CollapsiblePanel" />
            return this.appendHeader($(domElem("span")).text(headerText));
        },

        appendHeader: function (element) {
            /// <summary>Appends the specified HTML, DOM element or jQuery object to the
            /// header section of the CollapsiblePanel</summary>
            /// <param name="header" type="Selector">Content to append to the header section</param>
            /// <returns type="CollapsiblePanel" />
            this._header.append(element);
            return this;
        },

        appendContent: function (element) {
            /// <summary>Appends the specified content to the display content of the control</summary>
            /// <param name="content" type="Selector">This might be a jQuery selector or function.
            /// If a function is provided, that function will be executed whenever collapse icon is clicked.
            /// The function should return a content</param>
            /// <returns type="CollapsiblePanel" />

            var options = this._options, content;

            if ($.isFunction(element)) {
                if (options.collapsed) {
                    // If the specified content is a function, we'll add it to the internal
                    // list to be executed later when the collapse icn is clicked
                    if (!this._dynamicContents) {
                        this._dynamicContents = [];
                    }

                    this._dynamicContents.push(element);
                }
                else {
                    content = element.call(this);
                }
            }
            else {
                content = element;
            }

            if (content) {
                this._content.append(content);
                this._element.removeClass("collapsed");
            }

            return this;
        },

        onExpandCollapseClick: function (e) {
            var i, len, options = this._options,
                $icon, $content = this._content,
                contents = this._dynamicContents, contentFunc;

            // Changing the state of the control
            options.collapsed = !options.collapsed;

            // Changing the icon of the control
            $icon = $(e.target);
            $icon.removeClass(options.iconCollapseCss + " " + options.iconExpandCss);
            $icon.addClass(options.collapsed ? options.iconCollapseCss : options.iconExpandCss);

            // Working on the content
            if (options.collapsed) {
                // Hiding the content
                $content.hide();

                if (options.expandCss) {
                    this._element.removeClass(options.expandCss);
                }

                // TODO: Fire onCollapsed event
            }
            else {
                // Checking whether any dynamic content is remained to add to the content
                if (contents) {
                    for (i = 0, len = contents.length; i < len; i++) {
                        contentFunc = contents[i];
                        if ($.isFunction(contentFunc)) {
                            // Running the function and adding the returning content to the
                            // display content of the control
                            $content.append(contentFunc.apply());
                        }
                    }
                    // Resetting inner content list
                    this._dynamicContents = null;
                }

                // Showing the content
                $content.show();

                if (options.expandCss) {
                    this._element.addClass(options.expandCss);
                }

                // TODO: Fire onExpanded event
            }

            return false;
        }
    });


    function AjaxPanel(options) {
        this.baseConstructor.call(this, $.extend({
            cssClass: "ajax-panel",
            replaceContent: true,
            cache: true
        }, options));
    }

    AjaxPanel.extend({
        _typeName: "tfs.ajaxPanel"
    });

    AjaxPanel.inherit(Controls.BaseControl, function () {
        return {
            _cancelable: false,
            initialize: function () {
                var url = this._options.url;
                this._base();
                if (url) {
                    this.beginLoad(url, this._options.urlParams, this._options.success, this._options.error);
                }
            },
            _cancelPendingLoad: function () {
                if (this._cancelable) {
                    this._cancelable.cancel();
                    this._cancelable = null;
                }
            },
            _dispose: function () {
                this._cancelPendingLoad();
                this._base();
            },
            beginLoad: function (url, params, callback, errorcallback) {
                var self = this, cancelable, statusIndicator;

                Diag.logTracePoint("AjaxPanel.beginLoad.start");

                this._cancelPendingLoad();

                cancelable = new Core.Cancelable(this);
                this._cancelable = cancelable;

                this._element.empty();

                if (this._options.showStatusIndicator !== false) {
                    statusIndicator = StatusIndicator.createIn(this._element, { center: true, imageClass: "big-status-progress", message: Resources.Loading });
                    statusIndicator.start();
                }

                Ajax.getHTML(url, params, cancelable.wrap(function (content) {
                    self.onLoadCompleted(content);

                    if ($.isFunction(callback)) {
                        callback.call(self, content);
                    }
                }), cancelable.wrap(function (error) {
                    var handled;
                    if ($.isFunction(errorcallback)) {
                        handled = errorcallback.call(self, error) === true;
                    }

                    self.onLoadError(error, handled);
                }), {
                    cache: this._options.cache
                });
            },
            onLoadCompleted: function (content) {
                if (this._element && this._options.replaceContent) {
                    this._element.empty();
                    this._element.html(content);

                    Controls.Enhancement.ensureEnhancements(this._element);
                }

                Diag.logTracePoint("AjaxPanel.beginLoad.complete");
            },
            onLoadError: function (error, handled) {
                if (!handled) {
                    this.showError(error);
                }

                Diag.logTracePoint("AjaxPanel.beginLoad.complete");
            },
            showError: function (error) {
                this._element.empty();
                $(domElem("div", "error")).appendTo(this._element).text(getErrorMessage(error));
            }
        };
    }());


    function Dialog(options) {
        /// <summary>Creates a new dialog with the provided options</summary>
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "dialog",
            position: "center",
            widthPct: 0.8,
            heightPct: 0.8,
            modal: true,
            draggable: true,
            resizable: true,
            autoOpen: false,
            dynamicSize: true,
            disposeOnClose: true
        }, options));

        this._resizeDelegate = delegate(this, this._onResize);
    }

    Dialog.extend({
        _typeName: "tfs.dialog",
        _dialogActionInProgress: false,
        beginExecuteDialogAction: function (actionDelegate) {
            /// <summary>
            ///     This should be used in cases where you don't want the user to execute more than 1 particular action involving a Dialog
            ///     in parallel. For example, clicking a link that does some server processing before opening a dialog. On slow connections
            ///     the user may be able to click the link several times before the first dialog ever opens.
            /// </summary>
            /// <param name="actionDelegate" type="Function">
            ///     The function to execute which will involve initializing a Dialog. It takes a single optional
            ///     paramater which is a cancellation routine. Call this when you encounter a situation (such as an error)
            ///     where you wish to cancel the operation and have subsequent dialog actions execute without being blocked.
            /// </param>
            Diag.assertParamIsFunction(actionDelegate, "actionDelegate");

            if (!Dialog._dialogActionInProgress) {
                Dialog._dialogActionInProgress = true;
                actionDelegate(function () {
                    Dialog._dialogActionInProgress = false;
                });
            }
        },
        create: function (options) {
            var content = (options && options.content) || $("<div />");

            if (options && options.contentText) {
                content.text(options.contentText);
            }

            return this.enhance(content, options);
        },
        show: function (options) {
            return this.create($.extend({ autoOpen: true }, options));
        }
    });

    Dialog.inherit(AjaxPanel, {
        _title: null,
        _dialogResult: null,
        _resizeDelegate: null,
        _progressElement: null,

        initialize: function () {
            this._base();

            var self = this,
                $window = $(window),
                dialogOptions = $.extend({}, this._options);

            this._bind("dialogdragstart", function (e) {
                notificationService.fire("dialog-move", self);
            });

            if (this._options.attachResize) {
                this._bind("dialogresizestop", delegate(this, this.onDialogResize));
            }

            dialogOptions.title = this.getTitle().htmlEncode();
            dialogOptions.close = function (e) {
                if (self._options.close) {
                    self._options.close.call(this, e);
                }

                self.onClose(e);
            };

            dialogOptions.open = function (e) {
                if (self._options.open) {
                    self._options.open.call(this, e);
                }

                self.onOpen(e);
            };

            if (dialogOptions.dynamicSize) {
                dialogOptions.width = $window.width() * dialogOptions.widthPct;
                dialogOptions.height = $window.height() * dialogOptions.heightPct;

                notificationService.attachEvent("window-resize", this._resizeDelegate);
                this._bind("dialogclose", function (e, ui) {
                    notificationService.fire("dialog-close", self);
                    notificationService.detachEvent("window-resize", self._resizeDelegate);
                });
            }

            // Make sure the dialog is not taller or wider than the screen. In that case, the user is stuck
            // with no way to scroll.
            if (dialogOptions.height && !isNaN(dialogOptions.height)) {
                dialogOptions.height = Math.min(dialogOptions.height, $window.height());
            }
            if (dialogOptions.width && !isNaN(dialogOptions.width)) {
                dialogOptions.width = Math.min(1200, dialogOptions.width, $window.width());
            }

            if (!dialogOptions.url) {
                Diag.logTracePoint("Dialog.initialize.complete");
            }

            this._element.dialog(dialogOptions);

            this._element.keydown(function (event) {
                var button,
                    buttonOptions;
                if (event.keyCode === $.ui.keyCode.ENTER) {
                    if (!event.shiftKey && $(event.target).is("input[type='text']")) {
                        // if only ENTER is hit in a text <input>, which all our text input
                        // controls (including the combobox) "inherit" from
                        if (self._options.defaultButton) {
                            button = self._options.buttons[self._options.defaultButton];
                            if (button) {
                                // NOTE: There was a regression in jquery UI 1.8.14, .button("option", "disabled")
                                // does not return a boolean value by default anymore (because the option is undefined)
                                buttonOptions = $("#" + button.id).button("option");
                                if (!buttonOptions.disabled) {
                                    button.click.apply(this, [event]);
                                }
                            }
                            return false;
                        }
                    }
                }
            });

            // This is necessary for high contrast as the close icon is not visible
            this._element.dialog("option", "closeText", Resources.CloseButtonLabelText);

            // Allow other dialog actions to be executed
            Dialog._dialogActionInProgress = false;
        },

        onLoadCompleted: function (content) {
            this._base(content);

            // jQuery Dialog tries to set the focus initially but for the cases where
            // the content is loaded async, we need another attempt to set the focus
            // on the new content
            this.setInitialFocus();

            Diag.logTracePoint("Dialog.initialize.complete");
        },

        setInitialFocus: function () {
            /// <summary>Tries to set the focus using the specified or default selector</summary>
            var focusSelector = this._options.initialFocusSelector || ":visible:tabbable";
            this._element.find(focusSelector).first().focus();
        },

        setFormFocusDelayed: function ($field) {
            /// <summary>Sets focus on the first enabled input element in the dialog. </summary>
            /// <param name="field" type="Object">The field to set focus.</param>

            Diag.assertParamIsObject($field, "$field");

            // NOTE:
            // It's noticed that after appending the form elements to DOM, there's
            // a delay for browser to render the UI.
            // This behavior is also not consistent among browsers. Specifying a
            // 300 milli-sec delay is our workaround to make sure the default focus
            // (and thus the tabbing) behavior is consistent among supported ones.
            Core.delay(this, 300, function () {
                $field.eq(0)
                    .focus()
                    .select();
            });
        },

        _updateTitle: function () {
            if (this._initialized && this._element) {
                // We make the element check here because the dialog might be disposed when the
                // title is set asynchronously.
                this._element.dialog("option", "title", this.getTitle().htmlEncode());
            }
        },

        setTitle: function (title) {
            this._title = title;
            this._updateTitle();
        },

        getTitle: function () {
            return this._title || this._options.title || this._element.attr("title") || "";
        },

        getDialogResult: function () {
            return this._dialogResult;
        },

        setDialogResult: function (dialogResult) {
            this._dialogResult = dialogResult;
        },

        _onResize: function (e) {
            var $window = $(window),
                options = this._options;

            // Resetting width and height
            this._element.dialog("option", {
                width: $window.width() * options.widthPct,
                height: $window.height() * options.heightPct
            });

            // We need to set the position option separately again because
            // jQuery UI dialog tries to resize the dialog when width or height
            // is set in the options. This prevents the dialog to be repositioned centered.
            this._element.dialog("option", { position: options.position });
        },

        show: function () {
            this._element.dialog("open");
        },

        onOpen: function (e) {
            var progressContainer;

            if (!this._options || this._options.hasProgressElement !== false) {
                progressContainer = $("<div />")
                    .addClass("ui-dialog-titlebar-progress-container")
                    .appendTo(this._element.parent().find(".ui-dialog-titlebar"));

                this._progressElement = $("<div />")
                    .addClass("ui-dialog-titlebar-progress-element")
                    .appendTo(progressContainer);

                TFS.globalProgressIndicator.registerProgressElement(this._progressElement);
            }

            notificationService.fire("dialog-open", this);
        },

        onClose: function (e) {
            var dialogToFocus;

            notificationService.fire("dialog-close", this);

            if (this._progressElement) {
                TFS.globalProgressIndicator.unRegisterProgressElement(this._progressElement);
            }

            if (this._options.disposeOnClose) {
                this.dispose();
            }

            // Try to focus on topmost dialog
            dialogToFocus = $('.ui-dialog').slice(-1);
            if (dialogToFocus.length) {
                dialogToFocus.focus();
            }
        },

        close: function () {
            this._element.dialog("close");
        },

        onDialogResize: function (e) {
        }
    });

    function ModalDialog(options) {
        /// <summary>Creates a new modal dialog with specified options.By default, it has ok and cancel buttons</summary>
        var id = (options && options.id) || Controls.getId();

        this.baseConstructor.call(this, $.extend({
            coreCssClass: "dialog modal-dialog",
            modal: true,
            dynamicSize: false,
            width: 500,
            height: "auto",
            defaultButton: "ok",
            buttons: {
                "ok": {
                    id: "ok",
                    text: (options && options.okText) || Resources.ModalDialogOkButton,
                    click: delegate(this, this.onOkClick),
                    disabled: "disabled"
                },
                "cancel": {
                    id: "cancel",
                    text: (options && options.cancelText) || Resources.ModalDialogCancelButton,
                    click: delegate(this, this.onCancelClick)
                }
            }
        }, options));
    }

    ModalDialog.extend({
        _typeName: "tfs.modaldialog"
    });

    ModalDialog.inherit(Dialog, {

        initialize: function () {
            this._bind("buttonStatusChange", delegate(this, this.onButtonStatusChange));
            this._bind("resultReady", delegate(this, this.onResultReady));

            this._base();
        },

        updateOkButton: function (enabled) {
            if (this._element) {
                this._element.trigger("buttonStatusChange", { enabled: enabled === true });
            }
        },

        processResult: function (result) {
            var callback = this._options.okCallback;

            if (result) {
                if ($.isFunction(callback)) {
                    callback.call(this, result);
                    TFS.Diag.logTracePoint(this.getTypeName() + ".callback-complete");
                }

                this.close();
            }
        },

        onOkClick: function (e) {
            TFS.Diag.logTracePoint(this.getTypeName() + ".OkClicked");
            this.processResult(this.getDialogResult());
        },

        onResultReady: function (e, args) {
            this.processResult(args);
        },

        onCancelClick: function (e) {
            if ($.isFunction(this._options.cancelCallback)) {
                this._options.cancelCallback();
            }
            this.close();
        },

        onButtonStatusChange: function (e, args) {
            // If no button is specified, trying to change the status of OK button
            this._element.siblings(".ui-dialog-buttonpane").find("#" + (args.button || "ok")).button("option", "disabled", args.enabled ? "" : "disabled");
        }
    });

    function ConfirmationDialog(options) {
        this.baseConstructor.call(this, $.extend({
            width: 400,
            height: 200,
            resizable: false,
            cssClass: 'confirmation-dialog'
        }, options));
    }
    ConfirmationDialog.inherit(ModalDialog, {
        $errorContainer: null,
        initialize: function () {
            this._base();

            this.$errorContainer = $(domElem('div')).addClass('confirmation-dialog-error').prependTo(this._element).hide();

            this.updateOkButton(true);
        },
        _onSuccess: function () {
            this._fire('resultReady', true);
            var callback = this._options.successCallback;
            if ($.isFunction(callback)) {
                callback.apply(this, arguments);
                TFS.Diag.logTracePoint(this.getTypeName() + ".callback-complete");
            }
        },
        _onError: function (error) {
            this.$errorContainer.text(error.message).show();
        },
        onOkClick: function () {
            this.updateOkButton(false);
            this._base();
        }
    });

    function ToolbarItem(options) {
        this._options = $.extend({}, options);
        this._init();
    }

    ToolbarItem.prototype = {
        _options: null,
        _element: null,

        _init: function () {
        },

        appendTo: function (container) {
            this._element.appendTo(container);
        }
    };

    function ToolbarButton(options) {
        this.baseConstructor.call(this, $.extend({}, options));
    }

    ToolbarButton.extend({
        create: function (name, command) {
            var result;
            switch (command) {
                case "fore-color":
                case "back-color":
                    // Create a different kind of toolbar item like drop down
                    break;
                default:
                    result = new ToolbarButton({ name: name, command: command });
                    break;
            }
            return result;
        }
    });

    ToolbarButton.inherit(ToolbarItem, {
        _init: function () {
            this._base();

            this._element = $(domElem("span", "richeditor-toolbar-button"))
                .addClass("richeditor-toolbar-" + this._options.command)
                .attr("title", this._options.name)
                .attr("tabindex", 0)
                .attr("unselectable", "on")
                .hover(function () {
                    $(this).addClass("richeditor-toolbar-hover");
                },
                function () {
                    $(this).removeClass("richeditor-toolbar-hover");
                })
                .bind("click", delegate(this, this._onClick))
                .bind("keydown", delegate(this, this._onKeyDown));
        },

        _onClick: function () {
            // Firing toolbarButtonClick event so that, rich editor can handle toolbar command itself
            this._element.trigger("toolbarButtonClick", { command: this._options.command });
            return false;
        },

        _onKeyDown: function (e) {
            if (e.keyCode === TFS.UI.KeyCode.ENTER || e.keyCode === TFS.UI.KeyCode.SPACE) {
                return this._onClick();
            }
        }
    });

    function ToolbarButtonGroup(options) {
        this.baseConstructor.call(this, $.extend({}, options));
        this._buttons = [];
    }

    ToolbarButtonGroup.extend({
        create: function (name, createCallback) {
            var result = new ToolbarButtonGroup({ name: name });

            switch (name) {
                case "fontstyle":
                    result.addButton(ToolbarButton.create(Resources.EditorBold, "bold"));
                    result.addButton(ToolbarButton.create(Resources.EditorItalic, "italic"));
                    result.addButton(ToolbarButton.create(Resources.EditorUnderline, "underline"));
                    break;
                case "list":
                    result.addButton(ToolbarButton.create(Resources.EditorBulletedList, "insertunorderedlist"));
                    result.addButton(ToolbarButton.create(Resources.EditorNumberedList, "insertorderedlist"));
                    break;
                case "indentation":
                    result.addButton(ToolbarButton.create(Resources.EditorDecreaseIndent, "outdent"));
                    result.addButton(ToolbarButton.create(Resources.EditorIncreaseIndent, "indent"));
                    break;
                case "hyperlink":
                    result.addButton(ToolbarButton.create(Resources.EditorCreateLink, "createlink"));
                    result.addButton(ToolbarButton.create(Resources.EditorRemoveLink, "unlink"));
                    break;
                default:
                    if (createCallback && $.isFunction(createCallback)) {
                        createCallback(result);
                    }
                    break;
            }
            return result;
        }
    });

    ToolbarButtonGroup.inherit(ToolbarItem, {
        _buttons: null,

        _init: function () {
            this._base();
            this._element = $(domElem("div", "richeditor-toolbar-group"));
        },

        addButton: function (button) {
            this._buttons.push(button);
        },

        appendTo: function (container) {
            var i, len;

            this._base(container);
            this._element.empty();

            for (i = 0, len = this._buttons.length; i < len; i++) {
                this._buttons[i].appendTo(this._element);
            }
        }
    });

    function RichEditor(options) {
        /// <summary>Creates a new rich editor with the provided options</summary>
        this.baseConstructor.call(this, $.extend({
            height: "200px",
            buttonGroups: ["fontstyle", "list", "indentation", "hyperlink"]
        }, options));

        this._waitList = [];
    }

    RichEditor.extend({
        _typeName: "tfs.richeditor",
        INSERT_IMAGE_COMMAND: "insertImage",
        IMAGE_AUTOFIT_SCALE_FACTOR: 0.85
    });

    RichEditor.inherit(Controls.BaseControl, {
        _iframe: null,
        _window: null,
        _textArea: null,

        _isReady: false,
        _readyList: null,
        _editable: null,

        _toolbar: null,
        _navigator: null,
        _editor: null,
        _hasFocus: false,
        _explicitFocus: false,
        _keyDownInDocument: false,
        _customCommandHandlersMap: {},

        _originalValue: null,
        _currentValue: null,
        _needsRefocus: false,
        _refocusDelay: null,
        _attachDelay: null,

        _createIn: function (container) {
            var self = this;
            this._base(container);
            this._textArea = $("<textarea />").appendTo(this._element);


            if (this._options.id) {
                this._textArea
                .attr("id", this._options.id + "_txt")
                .bind("focus", function () {
                    self._textAreaFocus();
                });
            }

            this._decorate();
        },

        _enhance: function (element) {
            this._createElement();
            this._textArea = element;
            element.after(this._element);
            this._element.append(element);
            this._decorate();
        },

        _getToolbar: function () {
            if (!this._toolbar) {
                this._toolbar = this._createToolbar();
            }
            return this._toolbar;
        },

        _createToolbar: function () {
            var i, len,
                result,
                self = this,
                buttonGroups,
                customGroups;

            result = $(domElem("div", "richeditor-toolbar"))
                .hide()
                .prependTo(this._element);

            buttonGroups = this._options.buttonGroups || [];

            for (i = 0, len = buttonGroups.length; i < len; i++) {
                ToolbarButtonGroup.create(buttonGroups[i]).appendTo(result);
            }

            customGroups = this._options.customCommandGroups || [];

            for (i = 0, len = customGroups.length; i < len; i++) {
                ToolbarButtonGroup.create(customGroups[i].groupName, function (toolbarGroup) {
                    var buttons = customGroups[i].commands,
                        j, buttonCount;

                    for (j = 0, buttonCount = buttons.length; j < buttonCount; j += 1) {
                        // Dereference each command into an index by command name for easy retrieval during _executeCommand().
                        self._customCommandHandlersMap[buttons[j].command] = buttons[j].execute;

                        toolbarGroup.addButton(ToolbarButton.create(buttons[j].name, buttons[j].command));
                    }
                }).appendTo(result);
            }

            result.find(".richeditor-toolbar-button")
                .bind("focus", function (e) {
                    self._onFocusIn(e);
                })
                .bind("blur", function (e) {
                    self._explicitFocus = false;
                    self._onFocusOut(e);
                });

            if ($.browser.msie) {
                // IE loses selection of the document in iframe when focus is out. Thus, we disable getting focus
                // for toolbar buttons in IE so that focus is not lost when toolbar button is clicked
                TFS.UI.makeElementUnselectable(result[0]);
            }

            return result;
        },

        _showPanel: function (panel, opacity) {
            panel.stop(true, true);
            if (opacity === 0) {
                panel.fadeOut('slow');
            }
            else {
                panel.css("opacity", typeof (opacity) === "number" ? opacity : 1);
                panel.fadeIn('slow');
            }
        },

        _showToolbar: function (opacity) {
            log(verbose, "(richeditor) Showing toolbar (opacity=" + opacity + ")");
            this._showPanel(this._getToolbar(), opacity);
        },

        _getNavigator: function () {
            if (!this._navigator) {
                this._navigator = this._createNavigator().hide();
            }
            return this._navigator;
        },

        _createNavigator: function () {
            return $(domElem("div", "richeditor-navigator")).appendTo(this._element);
        },

        _showNavigator: function (opacity) {
            if (this._navigator) {
                log(verbose, "(richeditor) Showing link navigator (opacity=" + opacity + ")");
                this._showPanel(this._navigator, opacity);
            }
        },

        _decorate: function () {
            var iframe, options = this._options;

            this._element.addClass("richeditor-container");
            this._bind("toolbarButtonClick", delegate(this, this._onToolbarButtonClick));

            // Creating iframe
            iframe = $("<iframe />")
                .attr("frameBorder", "no")
                .appendTo($(domElem("div", "richeditor-editarea")).appendTo(this._element))
                .bind("load", delegate(this, this._initialize));

            // Setting iframe src and geting the reference
            iframe.attr("src", options.blankPageUrl);
            this._iframe = iframe[0];

            // There is no reason to display textarea because iframe replaces it.
            // Hiding textarea also improves the performance significantly especially when
            // the html content is huge. Setting a huge amount of text into a visible
            // textarea causes IE to freeze (See bug #743473)
            this._textArea.hide();
        },

        _initialize: function () {

            // Getting reference of iframe window
            this._window = this._iframe.contentWindow;
            $(this._window.document).ready(delegate(this, this._onDocumentReady));

            this._showToolbar(0.35);
        },

        _attachEvents: function () {
            /// <summary>Attaches necessary events to catch the changes if the control is enabled</summary>
            this._bind(this._iframe, "blur", delegate(this, this._onFocusOut));
            this._bind(this._iframe, "focus", delegate(this, this._onFocusIn));

            if (!$.browser.msie) {
                this._bind(this._window, "blur", delegate(this, this._onFocusOut));
                this._bind(this._window, "focus", delegate(this, this._onFocusIn));
            }

            this._bind(this._window.document, "mouseup", delegate(this, this._onMouseUp));
            this._bind(this._window.document, "keyup", delegate(this, this._onKeyUp));
            this._bind(this._window.document, "keydown", delegate(this, this._onKeyDown));
            this._bind(this._window.document, "keypress", delegate(this, this._onKeyPress));
            this._bind(this._window.document, "input", delegate(this, this._onInput));

            this._bind(this._window.document, "dblclick", delegate(this, this._onDblClick));

            if ($.isFunction(this._options.change)) {
                this._bind(this._textArea, "change", this._options.change);
            }
        },

        _detachEvents: function () {
            this._unbind(this._iframe, "blur");
            this._unbind(this._iframe, "focus");

            if (!$.browser.msie) {
                this._unbind(this._window, "blur");
                this._unbind(this._window, "focus");
            }

            // double click is a read only handler so there is no need to unbind it here.

            this._unbind(this._window.document, "mouseup");
            this._unbind(this._window.document, "keyup");

            if ($.isFunction(this._options.change)) {
                this._unbind(this._textArea, "change");
            }
        },

        _onDblClick: function (e) {
            var $src = $(e.srcElement);

            // The only double-click gesture that makes sense in the rich editor is to launch
            // inline images in a new tab/window.  Make sure that we only take action on images.

            // for src url with starting with data use alt to open attachment
            if ($src.is('img')) {
                var srcUrl = $src.attr("src");
                var altUrl = $src.attr("alt");

                var url = srcUrl.toLowerCase();

                if (url.indexOf("data") == 0) {
                    url = altUrl.toLowerCase();
                }
                this._window.open(url, "_blank");
            }
        },

        _onDocumentReady: function () {
            this._isReady = true;
            this._window.document.body.innerHTML = this._textArea.val();
            this._trySettingWaterMark(this._textArea.val());
            this._setEditable(this._options.enabled !== false);
            this._processReadyList();
        },

        _trySettingWaterMark: function (val) {
            if (this.isEmpty(val) && this._options.waterMark) {
                this._window.document.body.innerHTML = "<i class='watermark'>" + this._options.waterMark + "</i>";
                this._hasWaterMark = true;
            }
            else {
                this._hasWaterMark = false;
            }
        },

        _clearWaterMark: function () {
            if (this._hasWaterMark) {
                this._window.document.body.innerHTML = "";
                this._hasWaterMark = false;
            }
        },

        _textAreaFocus: function () {
            this._iframe.focus();
        },

        _onFocusIn: function (e) {
            if (!this._explicitFocus) {
                if (e && e.currentTarget) {
                    log(verbose, "(richeditor) focus in " + e.currentTarget.nodeType);
                }
                this._showToolbar();
            }
            this._hasFocus = true;
            this._clearWaterMark();
        },

        _onFocusOut: function (e) {
            Core.delay(this, 250, function () {
                if (!this._explicitFocus && !this._hasFocus) {
                    this._showToolbar(0.35);
                    this._showNavigator(0);
                }
                else {
                    this._explicitFocus = false;
                }
            });
            this._hasFocus = false;
            this._checkModified(e);
            this._trySettingWaterMark(this.getValue());
        },

        _onMouseUp: function (e) {
            if (this._options.fireOnEveryChange === true) {
                this._checkModified();
            }

            this._checkLinkNode();
        },

        _onKeyDown: function (e) {
            // Raise event on RichEditor control
            this._element.trigger(e);

            this._keyDownInDocument = true;
        },

        _onKeyPress: function (e) {
            // Raise event on RichEditor control
            this._element.trigger(e);
        },

        _onKeyUp: function (e) {
            if (this._options.fireOnEveryChange === true) {
                this._checkModified();
            }

            this._checkLinkNode();
            this._keyDownInDocument = false;

            // Raise event on RichEditor control
            this._element.trigger(e);
        },

        _onInput: function (e) {
            // Keystrokes will be handled by keydown/keyup. Handle this special case
            // where input is changing with no keypress. This can happen in webkit browsers
            // if you trigger the Undo action (from the menu or press ctrl-Z with focus outside this field)
            if (!this._keyDownInDocument) {
                if (this._options.fireOnEveryChange === true) {
                    this._checkModified();
                }
            }
        },

        _onToolbarButtonClick: function (e, args) {
            log(verbose, "(richeditor) toolbar click" + this._window.document.readyState);

            // We need to set this only for non-IE browsers because in IE case, window doesn't lose focus
            // when toolbar button is clicked because we made toolbar unselectable for IE
            this._explicitFocus = !$.browser.msie && this._hasFocus;
            this._executeCommand(args);

            if (this._options.fireOnEveryChange === true) {
                this._checkModified();
            }
        },

        _getNodeUnderCaret: function (tagName) {
            var node, resultNode = null,
                range = this._getTextRange();

            if (range) {
                if (range.parentElement) {
                    node = range.parentElement();
                }
                else if (range.commonAncestorContainer) {
                    node = range.commonAncestorContainer.nodeType === 3 ? range.commonAncestorContainer.parentNode : range.commonAncestorContainer;
                }

                if (node) {
                    resultNode = this._getNodeAncestor(node, tagName);
                }
            }
            return resultNode;
        },

        _getNodeAncestor: function (node, tagName) {
            /// <summary>Finds the node in the ancestors with the specified tag name</summary>

            // Tag names will always be in caps.
            tagName = tagName.toUpperCase();

            // Walk up the DOM until we find the node with the provided tag name
            while (node && node.tagName !== "BODY") {
                if (node.tagName === tagName) {
                    return node;
                }
                node = node.parentNode;
            }
            return null;
        },

        _getTextRange: function () {
            /// <summary> Gets a W3C Range or Microsoft TextRange object depending on the running browser.
            /// These object types are completely incompatible, so the caller must branch
            /// on platform or simply compare for equality.
            /// </summary>
            if (this._window.getSelection) {
                // W3C Standard

                var selection = this._window.getSelection();
                if (selection.getRangeAt) {
                    // Most browsers
                    if (selection.rangeCount > 0) {
                        return selection.getRangeAt(0);
                    }

                    return null;
                }
                else {
                    // Safari
                    var range = this._window.document.createRange();
                    range.setStart(selection.anchorNode, selection.anchorOffset);
                    range.setEnd(selection.focusNode, selection.focusOffset);
                    return range;
                }
            }
            else if (this._window.document.selection) {
                // IE
                return this._window.document.selection.createRange();
            }
        },

        _checkLinkNode: function () {
            /// <summary>Checks whether the element under the caret is a link or not</summary>

            var that = this,
                linkNode,
                linkClickHandler = this._options.linkClickHandler;

            linkNode = this._getNodeUnderCaret("A");

            function updateNavigator(href) {
                var $link,
                    linkText,
                    linkTextLength;

                if (href && !that._disposed) {

                    linkText = linkNode.href;
                    linkTextLength = linkText.length;
                    if (linkTextLength > 30) {
                        linkText = linkText.substring(0, 15) + "..." + linkText.substring(linkTextLength - 10, linkTextLength);
                    }

                    log(verbose, "(richeditor) Displaying link navigator");
                    $link = $("<a/>")
                        .attr({
                            target: "_blank",
                            href: href,
                            title: linkNode.href
                        })
                        .text(linkText)
                        .click(function (e) {
                            var handlerResult;
                            if ($.isFunction(linkClickHandler)) {
                                handlerResult = linkClickHandler(e);
                            }

                            if (handlerResult !== false) {
                                TFS.Host.ActionManager.performAction(TFS.Host.CommonActions.ACTION_WINDOW_OPEN, {
                                    url: href,
                                    target: "_blank"
                                });
                            }

                            return false;
                        })
                        .focus(function () {
                            $(this).addClass("focus");
                            that._explicitFocus = true;
                        })
                        .blur(function () {
                            $(this).removeClass("focus");
                            that._explicitFocus = false;
                            that._onFocusOut();
                        });

                    that._getNavigator()
                            .empty()
                            .append($("<span/>").text(Resources.NavigateTo))
                            .append(" ")
                            .append($link);

                    that._showNavigator();
                } else {
                    log(verbose, "(richeditor) Hiding link navigator");
                    that._showNavigator(0);
                }
            }

            if (linkNode) {
                TFS.Host.urlHelper.beginTranslateUrl(linkNode.href, this._options, updateNavigator, function (error) {
                    updateNavigator(null);
                });
            } else {
                updateNavigator(null);
            }
        },

        _checkModified: function (e) {
            /// <summary>Checks whether the value of the control is changed or not</summary>
            var newValue, value, fireEvent, isFocusOutEvent = e && e.type === 'blur';

            newValue = this.getValue();
            if (this._currentValue !== newValue || this._options.fireOnEveryChange) {

                // Fire change if transitioning from clean to dirty (or vice versa)
                fireEvent = newValue === this._originalValue ^ this._currentValue === this._originalValue;
                this._currentValue = newValue;

                this._fireChange(this._textArea, !fireEvent);

                // In IE, subsequent keypresses after the dirty/clean transition are
                // not captured until after a focusout/focusin
                if ($.browser.msie && fireEvent) {
                    this._needsRefocus = true;
                    log(TFS.Diag.LogVerbosity.Info, 'needs focus');
                }
            }

            if (this._needsRefocus) {
                this._tryRefocus(isFocusOutEvent);
            }
        },

        // This will try to refocus the element to enable the undo stack.
        // The first time this is called, the refocus will start.
        // Subsequent times this is called, nothing will happen if the refocus is still happening.
        // If the refocus is done, subsequent calls will check to see if the refocus is completed
        // by querying the undo command.  If the undo command is enabled, then we have refocused
        // successfully and we no longer need to refocus, so set _needsRefocus to false.
        // Sequence of events:
        //     1. Transition from clean to dirty (or dirty to clean)
        //     2. Start refocus
        //     3. Subsequent keypresses will try to refocus, but nothing will happen
        //     4. Refocus finishes, undo stack is enabled
        //     5. Next keypress will be captured by undo stack, and try to refocus,
        //            which sets _needsRefocus=false since the undo stack is enabled
        //     6. Subsequent keypresses will be captured by undo stack and not try to refocus
        _tryRefocus: function (isFocusOutEvent) {
            if (isFocusOutEvent) {
                // No longer need to refocus
                this._needsRefocus = false;
                if (this._refocusDelay) {
                    // Cancel pending refocus on focusout event
                    this._refocusDelay.cancel();
                    this._refocusDelay = null;
                }
            }
            else if (!this._refocusDelay) {
                // This delay is limited by response time in a new wit form and switching between two controls
                this._refocusDelay = Core.delay(this, 200, function () {
                    // This needs to be delayed because the undo command takes time to update after a refocus
                    var canUndo = this._iframe.contentDocument.queryCommandEnabled("Undo");
                    if (canUndo) {
                        log(TFS.Diag.LogVerbosity.Info, 'refocused');
                        this._needsRefocus = false;
                    }
                    else {
                        log(TFS.Diag.LogVerbosity.Info, 'refocusing');
                        this._refocus();
                    }
                    this._refocusDelay = null;
                });
            }
        },

        // This will refocus the control by focusing on another element,
        // and then back into the iframe.  Use this when a dom manipulation occurs
        // and you want to begin capturing keystrokes with the undo stack.
        _refocus: function () {
            if (this._attachDelay) {
                this._attachDelay.cancel();
                this._attachDelay = null;
            }
            // Temporarily disable focusout handler
            this._detachEvents();

            // Trigger focusout
            this._element.focus();

            // Give focus back
            this._iframe.contentWindow.focus();

            // Trigger focusin to show toolbar
            this._onFocusIn();

            // Re-enable focusout handler
            this._attachDelay = Core.delay(this, 100, function () {
                // This needs to be delayed or else focusout event will bubble up
                // after handler is attached, causing toolbar to disappear
                this._attachEvents();
                this._attachDelay = null;
            });
        },

        _executeCommand: function (commandInfo) {
            var command = commandInfo.command,
                provideUserInterface = null,
                args = commandInfo.args || null,
                link,
                url,
                customHandler,
                that = this;

            if (command === "createlink") {

                if ($.browser.msie) {
                    provideUserInterface = true;
                }
                else {
                    link = this._getNodeUnderCaret("a");
                    url = prompt(Resources.EditorEnterAddress, (link) ? link.href : "http://");

                    if ((url) && (url !== "")) {
                        args = url;
                    }
                }

                this._explicitFocus = false;
            }

            if (!$.browser.msie) {
                this._window.document.execCommand("useCSS", false, false);
            }

            customHandler = this._customCommandHandlersMap[command];

            if ($.isFunction(customHandler)) {
                customHandler(commandInfo, that);
            }
            else {
                this._window.focus();
                this._window.document.execCommand(command, provideUserInterface, args);
            }
        },

        _setEditable: function (value) {
            if (this._editable !== value) {
                this._editable = value;
                if (value) {
                    this._window.document.body.contentEditable = "true";
                    this._attachEvents();
                }
                else {
                    this._window.document.body.contentEditable = "false";
                    this._detachEvents();
                }
            }
        },

        ready: function (fn) {
            var list = this._readyList;

            if (!this._isReady) {
                if (!list) {
                    list = this._readyList = [];
                }
                list.push(fn);
            }
            else if ($.isFunction(fn)) {
                fn.call(this);
            }
        },

        _processReadyList: function () {
            var self = this;
            if (this._readyList) {
                $.each(this._readyList, function () {
                    if ($.isFunction(this)) {
                        this.call(self);
                    }
                });

                delete this._readyList;
            }
        },

        _ensureControlReadiness: function () {
            if (!this._isReady) {
                throw Resources.RichEditorControlNotReadyWarning;
            }
        },

        setEnabled: function (value) {
            this._ensureControlReadiness();
            this._setEditable(value);
        },

        getValue: function () {
            var doc = this._window.document, value = "";

            this._ensureControlReadiness();

            if (!this._hasWaterMark && doc.body) {
                value = this._normalizeValue(doc.body.innerHTML);
            }

            return value;
        },

        _normalizeValue: function (value) {
            return this.isEmpty(value) ? "" : value;
        },

        isEmpty: function (value) {
            // browsers change the html so that empty string means one of the followings
            return value === "" || value === "<br>" || value === "<P>&nbsp;</P>" || value === "<p>&nbsp;</p>" || value === "<br>\r\n";
        },

        setValue: function (value) {
            this._ensureControlReadiness();

            var doc = this._window.document;

            if (doc.body) {

                doc.body.innerHTML = value;
                this._trySettingWaterMark(value);

                this._originalValue = this._currentValue = this._normalizeValue(doc.body.innerHTML);
            }
            else {
                this._originalValue = this._currentValue = this._normalizeValue(value);
            }
        },

        insertImage: function (url) {
            /// <summary>Inserts an image tag pointing to the specified url at the current caret position if possible.
            /// If the current caret position cannot be determined, the image tag is inserted at the editor root node.
            /// </summary>
            /// <param name="url" type="string">The url containing an image in which to link to the document. </param>
            /// <remarks>
            /// The browser won't fire the change event of the TEXTAREA by default when an image
            /// or other complex object is inserted.  Instead, we will forcibly mark the field as dirty
            /// once we have inserted the image tag.
            /// </remarks>

            Diag.assertParamIsString(url, "url");

            Diag.logTracePoint("RichEditor.InlineImage.insertion.start");

            this._ensureControlReadiness();
            this._window.focus();
            this._window.document.execCommand(RichEditor.INSERT_IMAGE_COMMAND, false, url);

            // NOTE: this is an expensive selector but since we are using the execCommand
            // infrastructure we don't have a dom element that refers to the image so we have
            // to go find it using the only natural key we have: the url.

            var $imgElement = $('img[src="' + url + '"]', this._window.document);

            // KLUDGE: The size of the image isn't known until it has finished rendering.  Wait
            // until then to adjust the extents to avoid overtaking the description field.

            $imgElement.bind("load", delegate(this, function (e) {

                // Setting the width here will respect the aspect ratio of the image
                // so there is no need to adjust the height.
                $imgElement.width(Math.min($imgElement.width(), $(this._window.document.body).width() * RichEditor.IMAGE_AUTOFIT_SCALE_FACTOR));

                // The browser won't fire the change event of the TEXTAREA by default when an image
                // or other complex object is inserted.  Instead, we will forcibly mark the field as dirty
                // once we have inserted the image tag.

                if ($.isFunction(this._options.change)) {
                    this._options.change();
                }

                Diag.logTracePoint("RichEditor.InlineImage.insertion.complete");
            }));
        }
    });


    function TreeView(options) {
        /// <summary>Creates new Grid Control</summary>
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "tree-view",
            showIcons: true,
            clickToggles: false,
            clickSelects: true,
            contextMenu: false,
            styleFocusElement: true,
            useEmptyFolderNodes: true
        }, options));

        this.rootNode = new Controls.TreeNode("root");
        this.rootNode.root = true;
        this._droppable = options.droppable;
        this._draggable = options.draggable;
        this._focusDelegate = delegate(this, this._onFocus);
        this._blurDelegate = delegate(this, this._onBlur);
    }

    TreeView.extend({
        _typeName: "tfs.treeView",
        NODE_DATA_NAME: "node-data",
        LEVEL_DATA_NAME: "node-level"
    });

    TreeView.inherit(Controls.BaseControl, function ($) {
        return {
            rootNode: null,
            _focusDelegate: null,
            _blurDelegate: null,
            _hasFocus: false,
            _selectedNode: null,
            _draggable: null,
            _droppable: null,
            _focusedNode: null,
            _popupMenu: null,

            initialize: function () {
                var that = this;
                this._base();

                this._bind("mouseover", function (e) {
                    var $target = $(e.target), $closest, $node;

                    $closest = $target.closest("div.node-content, div.node-context-menu");

                    if ($closest.length > 0) {
                        $node = $closest.closest("li.node");

                        that._setFocusElement($node);
                    }
                });

                this._bind("mouseout", function (e) {
                    that._setFocusElement(null);
                });

                this._bind("click", delegate(this, this._click));
                this._bind("contextmenu", delegate(this, this._onContextMenu));
                this._bind("keydown", delegate(this, this._onInputKeyDown));

                // Checking any nodes specified in the options
                if ($.isArray(this._options.nodes)) {
                    this.rootNode.addRange(this._options.nodes);
                }

                this._draw();

                TFS.Diag.logTracePoint("TreeView.initialize.complete");
            },
            _draw: function () {
                this._element.empty();
                this._drawNode(this.rootNode, this._element, -1);
            },
            _getNodeElement: function (node) {
                return this._element.find("#tfs_tnli" + node.id).eq(0);
            },
            _getNode: function ($element) {
                /// <summary>Get the node associated with the element</summary>
                /// <param name="$element" type="jQuery(LI)">The jQuery object wrapping the tree node's DOM element</param>
                /// <returns type="TreeNode" />
                return $element.data(TreeView.NODE_DATA_NAME);
            },
            getSelectedNode: function () {
                return this._selectedNode;
            },
            setSelectedNode: function (node) {
                if (this._selectedNode) {
                    this._selectedNode.selected = false;
                }

                this._selectedNode = node;
                if (node) {
                    node.selected = true;
                    this._fire("selectionchanged", { selectedNode: node });

                    this._expandNodeParents(node);
                }

                this._updateSelections();

            },
            focus: function () {
                if (this._selectedNode) {
                    this._getNodeElement(this._selectedNode).find('.node-link').focus();
                }
            },
            _expandNodeParents: function (node) {
                var n = node, root = this.rootNode, startNode, nodeElement;

                while (n.parent) {
                    n = n.parent;

                    if (n !== root && !n.expanded) {
                        n.expanded = true;
                        startNode = n;
                    }
                }

                if (startNode) {
                    startNode.expanded = false;
                    nodeElement = this._getNodeElement(startNode);
                    this._toggle(startNode, nodeElement);
                }
            },
            _updateSelections: function () {
                var that = this;
                this._element.find("li.node").each(function () {
                    var li = $(this), node;
                    node = that._getNode(li);
                    li.toggleClass("selected", node.selected);
                });
            },
            _drawNode: function (node, parentElement, level) {
                var li;

                if (node.root) {
                    li = parentElement;
                }
                else {
                    li = $("<li />").appendTo(parentElement);
                }

                this._updateNode(li, node, level);
            },
            _drawEmptyFolderNode: function (parentElement, level, text) {
                var node, li;

                li = $("<li />").appendTo(parentElement);
                node = new Controls.TreeNode(text);
                node.isEmptyFolderChildNode = true;
                node.config.css = "info";
                node.config.unselectable = true;
                node.noFocus = true;
                node.noContextMenu = true;

                this._updateNode(li, node, level);
            },
            _updateNode: function (li, node, level) {
                var link, div;

                li.addClass("node");

                if (!node.root) {
                    li.attr("title", node.title || node.text || "")
                      .attr("id", "tfs_tnli" + node.id);

                    // If the node is tagged as draggable and we have draggable options, apply them.
                    if (node.draggable && this._draggable) {
                        li.draggable(this._draggable);
                    }

                    // If the node is tagged as droppable and we have droppable options, apply them.
                    if (node.droppable && this._droppable) {
                        li.droppable(this._droppable);
                    }

                    if (node.selected) {
                        li.addClass("selected");
                    }

                    if (node.noFocus) {
                        li.addClass("nofocus");
                    }

                    if (node.folder) {
                        li.addClass("folder");
                    }

                    if (node.config && node.config.css) {
                        li.addClass(node.config.css);
                    }

                    if (this._options.contextMenu && !node.noContextMenu) {
                        li.append($(domElem("div", "node-context-menu icon")));
                    } else {
                        if (node.noContextMenu) {
                            li.append($(domElem("div", "node-no-context-menu")));
                        }
                    }

                    link = $("<a />").addClass("node-link");
                    if (node.link) {
                        link.attr("href", node.link);
                    }

                    if (!node.noFocus) {
                        link.attr("tabindex", 0);
                        this._bind(link, "focus", this._focusDelegate);
                        this._bind(link, "blur", this._blurDelegate);
                    }

                    div = $("<div />")
                    .addClass("node-content")
                    .css("padding-left", level * 12);

                    link.append(div);

                    if (!node.noTreeIcon) {
                        $(domElem("span", "node-img icon")).appendTo(div);
                    }

                    if (this._options.showIcons && node.icon) {
                        $("<span />")
                        .addClass(node.icon)
                        .appendTo(div);
                    }

                    div.append(node.text || "");

                    li.append(link);
                }

                // Set the node in the data for the list item so we can lookup
                // which node is associated with the element.
                li.data(TreeView.NODE_DATA_NAME, node);
                li.data(TreeView.LEVEL_DATA_NAME, level);

                if (node.hasChildren() || (node.folder && this._options.useEmptyFolderNodes)) {
                    this._drawChildren(node, li, level);
                }

                return div;
            },
            _drawChildren: function (node, nodeElement, level) {
                var ul, i, len;
                if (node.root || node.expanded) {
                    nodeElement.addClass("expanded");

                    ul = $(domElem("ul", "tree-children"))
                         .attr("id", "tfs_tnul" + node.id);

                    if (typeof level === "undefined") {
                        level = nodeElement.data(TreeView.LEVEL_DATA_NAME);
                    }

                    level = level + 1;

                    len = node.children.length;
                    if (len === 0) {
                        this._drawEmptyFolderNode(ul, level, node.emptyFolderNodeText || this._options.defaultEmptyFolderNodeText || Resources.NoItemsInThisFolder);
                    }
                    else {
                        for (i = 0; i < len; i++) {
                            this._drawNode(node.children[i], ul, level);
                        }
                    }

                    nodeElement.append(ul);
                }
                else {
                    nodeElement.addClass("collapsed");
                }
            },

            _click: function (e) {
                var $target = $(e.target), $closestNode = $target.closest("li.node");

                if ($closestNode.length > 0) {
                    this._setFocusElement($closestNode);

                    if ($target.closest("span.node-img").length > 0) {
                        return this._onToggle(e);
                    }
                    else if ($target.closest("div.node-content").length > 0) {
                        return this._itemClick(e);
                    }
                    else if ($target.closest("div.node-context-menu").length > 0) {
                        return this._onContextMenu(e);
                    }

                }
            },
            _onInputKeyDown: function (e) {
                /// <summary>Handle key down events (node selection & expansion)</summary>
                /// <param name="e" type="event">The keydown event</param>
                /// <returns type="Boolean" />
                var $target = $(e.target), $closestNode = $target.closest("li.node"), node;

                switch (e.keyCode) {
                    case keyCode.RIGHT:
                        return this._setNodeExpansion(this._getNode($closestNode), $closestNode, true);
                    case keyCode.LEFT:
                        return this._setNodeExpansion(this._getNode($closestNode), $closestNode, false);
                    case keyCode.ENTER:
                        // If no href associated with node-link, click the node
                        if (!$(".node-link", $closestNode).attr("href")) {
                            node = this._getNode($closestNode);
                            this.onItemClick(node, $closestNode);
                            return false;
                        }
                        break;
                    case 121: //F10 key
                        if (e.shiftKey) {
                            return this._onContextMenu();
                        }
                        else {
                            return;
                        }
                }
            },
            _onToggle: function (e) {
                var li = $(e.target).parents("li.node:first");
                this._toggle(this._getNode(li), li);
                return false;
            },
            _toggle: function (node, nodeElement) {
                var ul;
                if (node.folder || node.hasChildren()) {
                    nodeElement.removeClass("collapsed expanded");
                    if (node.expanded) {
                        ul = nodeElement.find("ul.tree-children:first");
                        ul.remove();
                        node.expanded = false;
                        nodeElement.addClass("collapsed");
                    }
                    else {
                        node.expanded = true;
                        this._drawChildren(node, nodeElement);
                    }

                    return true;
                }

                return false;
            },
            _setNodeExpansion: function (node, nodeElement, expand) {
                /// <summary>Ensure the tree node's expansion state is set to a particular value</summary>
                /// <param name="node" type="TreeNode">The tree node</param>
                /// <param name="nodeElement" type="jQuery(LI)">The element associated with the node</param>
                /// <param name="expand" type="Boolean">The desired expand state of the node - true = expanded, false = collapsed</param>
                /// <returns type="Boolean">true = the node's expansion state was changed, false otherwise</returns>
                if (node.expanded !== expand) {
                    return this._toggle(node, nodeElement);
                }
                return false;
            },
            removeNode: function (node) {
                this._getNodeElement(node).remove();
                node.remove();
            },
            updateNode: function (node) {
                var nodeElement, level;
                if (node.root) {
                    nodeElement = this._element;
                    level = -1;
                }
                else {
                    nodeElement = this._getNodeElement(node);
                    level = nodeElement.data(TreeView.LEVEL_DATA_NAME);
                    nodeElement.removeClass(); //will remove all classes
                }
                nodeElement.empty();
                this._updateNode(nodeElement, node, level);
            },
            _itemClick: function (e) {
                var li = $(e.target).closest("li.node"), node;

                node = this._getNode(li);

                return this.onItemClick(node, li, e);
            },
            onItemClick: function (node, nodeElement, e) {
                if (this._options.clickToggles) {
                    if (this._toggle(node, nodeElement)) {
                        return false;
                    }
                }

                if (this._options.clickSelects) {
                    this.setSelectedNode(node);
                }
            },

            _onContextMenu: function (e) {
                if (this._options.contextMenu) {
                    if (this._focusedNode && this._focusedNode.length > 0) {
                        this._showPopupMenu(this._getNode(this._focusedNode));
                        return false;
                    }
                }
            },

            _showPopupMenu: function (node) {
                var nodeContextMenu, option;
                if (node) {
                    nodeContextMenu = node.contextMenu;

                    if (typeof nodeContextMenu !== "undefined") {
                        if ($.isFunction(nodeContextMenu)) {
                            option = nodeContextMenu.call(this, node);
                        }
                        else {
                            option = nodeContextMenu;
                        }
                    }
                    else {
                        option = this._options.contextMenu;
                    }

                    this.onShowPopupMenu(node, option);
                }
            },

            onShowPopupMenu: function (node, options) {
                var items, nodeElement, escapeReceiver, menuPin;

                if (options && options.items && options.items.length) {
                    nodeElement = this._getNodeElement(node);
                    escapeReceiver = nodeElement.find("a.node-link, div.node-content[tabIndex]").first()[0];
                    menuPin = nodeElement.find("div.node-context-menu");

                    if (this._popupMenu) {
                        if (nodeElement.hasClass("context-menu-active")) {
                            this._popupMenu.popup(escapeReceiver, menuPin);

                            return;
                        }
                        else {
                            this._popupMenu.dispose();
                            this._popupMenu = null;
                        }
                    }

                    items = options.items.slice(0); //create a clone
                    items.sort(function (a, b) { return (a.rank || 9999) - (b.rank || 9999); });

                    // Creating the popup menu which will be displayed when the gutter is clicked
                    this._popupMenu = MenuControls.PopupMenu.createIn(nodeElement, {
                        align: options.align || "left-bottom",
                        executeAction: options.executeAction || options.clickHandler,
                        items: [{
                            childItems: items
                        }],
                        onActivate: function () {
                            nodeElement.addClass("context-menu-active");
                        },
                        onDeactivate: function () {
                            nodeElement.removeClass("context-menu-active");
                        },
                        onPopupEscaped: function () {
                            nodeElement.removeClass("context-menu-active");
                        },
                        getCommandState: options.getCommandState,
                        updateCommandStates: options.updateCommandStates,
                        arguments: options.arguments,
                        contextInfo: { item: node, menu: this._popupMenu }
                    });

                    // We need to make sure that the popup menu is the first element of this node
                    // because if there are a lot of child nodes, popup menu goes off screen and
                    // when it's being popped up, it tries to activate the focus element (which is off screen)
                    // and causes issues with the scroll
                    this._popupMenu.getElement().prependTo(nodeElement);

                    // Displaying the popup
                    // Grid set tries to set focus on container mouse down event with a timeout
                    // This behavior causes our popup menu item to close immediately since it loses focus.
                    // Lets popup our menu in another epoch
                    Core.delay(this, 10, function () {
                        this._popupMenu.popup(escapeReceiver, menuPin);
                    });
                }
            },

            enableFocusStyling: function (enabled) {
                /// <summary>Indicate whether the element that has focus should be styled differently.
                /// The current focus element will be updated to match the new preference</summary>
                /// <param name="enabled" type="Bool">true, if focus element should be styled.</param>

                if (this._options.styleFocusElement !== enabled) {
                    this._options.styleFocusElement = enabled;

                    // update styling to ensure it's consistent with the new styling status
                    this._setFocusElement(this._focusedNode);
                }
            },

            _setFocusElement: function (element) {
                var styleFocusElement = this._options.styleFocusElement;

                if (this._focusedNode && this._focusedNode.length) {
                    this._focusedNode.removeClass("focus");
                }

                this._focusedNode = element;

                if (styleFocusElement && this._focusedNode && this._focusedNode.length) {
                    if (!this._focusedNode.hasClass("nofocus")) {
                        this._focusedNode.addClass("focus");
                    }
                }
            },

            _onFocus: function (e) {
                var $closestNode = $(e.target).closest("li.node");

                if ($closestNode.length > 0) {
                    this._setFocusElement($closestNode);
                }

                this._element.toggleClass("focus", true);
                this._hasFocus = true;
            },

            _onBlur: function (e) {
                this._setFocusElement(null);
                this._element.toggleClass("focus", false);
                this._hasFocus = false;
            },

            getNodeFromElement: function (element) {
                /// <summary>Gets the node associated with the provided element.</summary>
                /// <param name="element" type="object">Element to get the node for.</param>
                /// <returns type="object" />
                Diag.assertParamIsNotNull(element, "element");

                var $element = $(element);

                return this._getNode($element);
            }
        };
    }(jQuery));

    function VerticalBreadCrumb(options) {
        /// <summary>Creates new VerticalBreadCrumb Control</summary>
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "vertical-bread-crumb"
        }, options));
    }

    VerticalBreadCrumb.extend({
        _typeName: "tfs.verticalBreadCrumb",
        NODE_DATA_NAME: "node-data"
    });

    VerticalBreadCrumb.inherit(Controls.BaseControl, function ($) {
        return {
            _node: null,
            _focusedNode: null,
            _ul: null,
            initialize: function () {
                var that = this;
                this._base();
                this._draw();

                this._bind("mouseover", function (e) {
                    var $target = $(e.target), $closest, $node;

                    //$closest = $target.closest("div.node-content");
                    $closest = $target.closest("div.node-content, div.node-context-menu");

                    if ($closest.length > 0) {
                        $node = $closest.closest("li.node");
                        that._setFocusElement($node);
                    }
                });

                this._bind("mouseout", function (e) {
                    that._setFocusElement(null);
                });

                this._bind("click", delegate(this, this._click));

                TFS.Diag.logTracePoint("VerticalBreadCrumb.initialize.complete");
            },
            setNode: function (node) {
                this._node = node;
                this._draw();
            },
            _setFocusElement: function (element) {
                if (this._focusedNode && this._focusedNode.length) {
                    this._focusedNode.removeClass("focus");
                }

                this._focusedNode = element;

                if (this._focusedNode && this._focusedNode.length) {
                    if (!this._focusedNode.hasClass("nofocus")) {
                        this._focusedNode.addClass("focus");
                    }
                }
            },
            _click: function (e) {
                var $target = $(e.target), $closestNode = $target.closest("li.node");

                if ($closestNode.length > 0) {
                    this._setFocusElement($closestNode);

                    if ($target.closest("div.node-context-menu").length > 0) {
                        return this._onContextMenu(e);
                    }
                }
            },
            _onContextMenu: function (e) {
                if (this._options.contextMenu) {
                    if (this._focusedNode && this._focusedNode.length > 0) {
                        this._showPopupMenu(this._focusedNode.data(VerticalBreadCrumb.NODE_DATA_NAME));
                    }
                }
            },

            _showPopupMenu: function (node) {
                var nodeContextMenu, option;
                if (node) {
                    nodeContextMenu = node.contextMenu;

                    if (typeof nodeContextMenu !== "undefined") {
                        if ($.isFunction(nodeContextMenu)) {
                            option = nodeContextMenu.call(this, node);
                        }
                        else {
                            option = nodeContextMenu;
                        }
                    }
                    else {
                        option = this._options.contextMenu;
                    }

                    this.onShowPopupMenu(node, option);
                }
            },

            onShowPopupMenu: function (node, options) {
                var items, nodeElement, escapeReceiver, menuPin;

                if (options && options.items && options.items.length) {
                    nodeElement = this._focusedNode;
                    escapeReceiver = nodeElement.find("div.node-content").first()[0];
                    menuPin = nodeElement.find("div.node-context-menu");

                    if (this._popupMenu) {
                        if (nodeElement.hasClass("context-menu-active")) {
                            this._popupMenu.popup(escapeReceiver, menuPin);

                            return;
                        }
                        else {
                            this._popupMenu.dispose();
                        }
                    }

                    items = options.items.slice(0); //create a clone
                    items.sort(function (a, b) { return (a.rank || 9999) - (b.rank || 9999); });


                    // Creating the popup menu which will be displayed when the gutter is clicked
                    this._popupMenu = MenuControls.PopupMenu.createIn(nodeElement, {
                        align: options.align || "left-bottom",
                        executeAction: options.executeAction || options.clickHandler,
                        items: [{
                            childItems: items
                        }],
                        onActivate: function () {
                            nodeElement.addClass("context-menu-active");
                        },
                        onDeactivate: function () {
                            nodeElement.removeClass("context-menu-active");
                        },
                        onPopupEscaped: function () {
                            nodeElement.removeClass("context-menu-active");
                        }
                    });

                    this._popupMenu.actionArguments = options.actionArguments || { item: node, menu: this._popupMenu };
                    // Setting action arguments
                    if ($.isFunction(options.getActionArguments)) {
                        this._popupMenu.actionArguments = options.getActionArguments.call(this, this._popupMenu.actionArguments);
                    }

                    // Displaying the popup
                    // Grid set tries to set focus on container mouse down event with a timeout
                    // This behavior causes our popup menu item to close immediately since it loses focus.
                    // Lets popup our menu in another epoch
                    Core.delay(this, 10, function () {
                        this._popupMenu.popup(escapeReceiver, menuPin);
                    });
                }
            },
            _draw: function () {
                var i, l, level = 0, index = 0;
                this._element.empty();
                this._ul = $("<ul />");

                if (this._node) {
                    if (this._node.parents) {
                        for (i = 0, l = this._node.parents.length; i < l; i++) {
                            this._drawNode(this._node.parents[i], level, index++);
                        }

                        if (l > 0) {
                            level++;
                        }
                    }

                    this._drawNode(this._node, level, index++);

                    level++;

                    if (this._node.children) {
                        for (i = 0, l = this._node.children.length; i < l; i++) {
                            this._drawNode(this._node.children[i], level, index++);
                        }
                    }
                }

                this._element.append(this._ul);
            },
            _drawNode: function (node, level, index) {
                var li, treeIcon, link, div;

                li = $("<li />")
                    .attr("title", node.title || node.text || "")
                    .addClass("node")
                    .appendTo(this._ul);

                if (index > 0) {
                    if (level === 0) {
                        li.addClass("up");
                    }
                    else {
                        li.addClass("corner");
                    }
                }

                if (node.selected) {
                    li.addClass("selected");
                }

                if (node.config && node.config.css) {
                    li.addClass(node.config.css);
                }

                if (this._options.contextMenu) {
                    $(domElem("div", "node-context-menu icon")).appendTo(li);
                }

                // The padding is calculated as 10 + level * 6.
                // 10 is the default indentation for all tree views (see TreeView _drawNode).
                // For vertical crumbs we want to align each level based on it's icon and use as little indentation as possible.
                // 6 pixels aligns the icon indentations from child to the parent breadcrumb.
                div = $("<div />").addClass('node-content').css("padding-left", 10 + (level * 6)).appendTo(li);

                treeIcon = $(domElem("span", "icon"))
                            .addClass("bread-crumb")
                            .appendTo(div);

                if (node.link) {
                    link = $("<a />")
                       .attr("href", node.link)
                       .addClass("node-link")
                       .appendTo(li)
                       .append(div);
                }
                else {
                    div.appendTo(li)
                    div.attr("tabIndex", 0);
                }

                div.append(node.text || "");

                li.data(VerticalBreadCrumb.NODE_DATA_NAME, node);
                this._updateNode(li, node);
            },
            _updateNode: function (li, node) {

            }
        };
    }(jQuery));


    function Splitter(options) {
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "splitter",
            vertical: true,
            fixedSide: "left"
        }, options));
    }

    Splitter.extend({
        _typeName: "tfs.splitter"
    });

    Splitter.inherit(Controls.BaseControl, function () {
        return {
            leftPane: null,
            rightPane: null,
            handleBar: null,
            _screenXY: "pageX",
            _cssPosProp: "left",
            _cssSizeProp: "width",
            _leftFix: true,
            _fixedSide: null,
            _fillSide: null,
            _deltaMultiplier: 1,
            _dragStart: false,
            _fixedSidePixels: null,
            _overlay: null,
            _ignoreWindowResize: false,
            expandState: null,
            _createElement: function () {
                this._base();
                if (this._options.vertical === false) {
                    this._element.addClass("horizontal");
                }
                else {
                    this._element.addClass("vertical");
                }

                if (this._options.expandState) {
                    this.expandState = this._options.expandState;
                    this._element.addClass(this.expandState === "left" ? "left-expand" : "right-expand");
                }

                if (this._options.fixedSide === "right") {
                    this._element.addClass("right-fix");
                }

                this.leftPane = $(domElem("div", "leftPane")).appendTo(this._element);
                this.handleBar = $(domElem("div", "handleBar")).appendTo(this._element);
                this.rightPane = $(domElem("div", "rightPane")).appendTo(this._element);

                this._attachEvents();
            },
            _enhance: function (element) {
                this._base(element);

                this._options.vertical = element.hasClass("vertical");

                this.leftPane = this._element.children(".leftPane");
                this.handleBar = this._element.children(".handleBar");
                this.rightPane = this._element.children(".rightPane");

                if (this._element.hasClass("left-expand")) {
                    this.expandState = "left";
                }
                else if (this._element.hasClass("left-expand")) {
                    this.expandState = "right";
                }

                if (this._element.hasClass("right-fix")) {
                    this._options.fixedSide = "right";
                }

                this._attachEvents();
            },
            initialize: function () {
                this._base();
                this._configureCssProps();
            },
            _configureCssProps: function () {
                var leftFix = this._leftFix = this._options.fixedSide === "left";

                if (leftFix) {
                    this._deltaMultiplier = 1;
                    this._fixedSide = this.leftPane;
                    this._fillSide = this.rightPane;
                }
                else {
                    this._deltaMultiplier = -1;
                    this._fixedSide = this.rightPane;
                    this._fillSide = this.leftPane;
                }

                if (this._options.vertical === false) {
                    this._screenXY = "pageX";
                    this._cssSizeProp = "width";
                    this._cssPosProp = leftFix ? "left" : "right";
                }
                else {
                    this._screenXY = "pageY";
                    this._cssSizeProp = "height";
                    this._cssPosProp = leftFix ? "top" : "bottom";
                }
            },
            _attachEvents: function () {
                var that = this;
                this._bind(this.handleBar, "mousedown", delegate(this, this._handleBarMouseDown));
                this._bind(this.handleBar, "dblclick", delegate(this, this._handleBarDoubleClick));
                this._bind(this.handleBar, "mouseover", function (e) { that.handleBar.addClass("hover"); });
                this._bind(this.handleBar, "mouseout", function (e) { that.handleBar.removeClass("hover"); });
                this._bind(this.handleBar, "selectstart", function () { return false; });
                this._bind($(window), "resize", delegate(this, this._onWindowResize));
            },
            _measureFixedSide: function () {
                this._fixedSidePixels = this._fixedSide[this._cssSizeProp]();
            },
            _handleBarMouseDown: function (e) {
                this._dragStart = {
                    pointer: e[this._screenXY],
                    originalSize: this._fixedSide[this._cssSizeProp]()
                };

                this._element.addClass("dragging");
                this._setupDragEvents();

                // Draw the overlay on handle bar mouse down. Otherwise (if we wait until the mouse move / delta change)
                // in some browsers (Chrome) text will start getting selected on the page
                this._ensureOverlay();
            },
            _setupDragEvents: function () {
                this._bind(window.document, "mousemove", delegate(this, this._documentMouseMove), true);
                this._bind(window.document, "mouseup", delegate(this, this._documentMouseUp), true);
            },
            _ensureOverlay: function () {
                if (!this._overlay) {
                    this._overlay = $(domElem("div", "overlay"));
                    this._element.append(this._overlay);

                    this._bind(this._overlay, "mousemove", delegate(this, this._documentMouseMove));
                    this._bind(this._overlay, "mouseup", delegate(this, this._documentMouseUp));
                    this._bind(this._overlay, "selectstart", function () { return false; });
                }
            },
            _removeOverlay: function () {
                if (this._overlay) {
                    this._overlay.remove();
                    this._overlay = null;
                }
            },
            _clearDragEvents: function () {
                this._removeOverlay();

                this._unbind(window.document, "mousemove");
                this._unbind(window.document, "mouseup");
            },
            _documentMouseMove: function (e) {
                var delta;

                if (this._dragStart) {
                    delta = e[this._screenXY] - this._dragStart.pointer;

                    if (delta) {
                        this._ensureOverlay();
                    }

                    this.handleBar.css(this._cssPosProp, +(this._dragStart.originalSize + this._deltaMultiplier * delta) + "px");

                    return false;
                }
            },
            _documentMouseUp: function (e) {
                var delta, changed;

                if (this._dragStart) {
                    delta = e[this._screenXY] - this._dragStart.pointer;

                    if (Math.abs(delta) > 1) {
                        changed = true;

                        if (this.expandState) {
                            this.removeExpand(true);
                        }

                        this._resize(this._dragStart.originalSize + this._deltaMultiplier * delta);
                    }

                    this._dragStart = null;
                    this._element.removeClass("dragging");
                    this._clearDragEvents();

                    if (changed) {
                        this._fire("changed", this);
                    }

                    return false;
                }
            },
            _onWindowResize: function () {
                if (!this._ignoreWindowResize) {
                    if (!this.expandState) {
                        this._resize(this._fixedSidePixels, true);
                    }
                }
            },
            _fireWindowResize: function () {
                try {
                    this._ignoreWindowResize = true;
                    $(window).trigger("resize");
                } finally {
                    this._ignoreWindowResize = false;
                }

            },
            _resize: function (newSize, suppressFireResize) {
                var elemSize, newSizeCss;
                if (newSize != null) {
                    if (newSize < 0) {
                        newSize = 0;
                    } else {
                        elemSize = this._element[this._cssSizeProp]() - this.handleBar[this._cssSizeProp]();
                        if (newSize > elemSize) {
                            newSize = elemSize;
                        }
                    }
                }

                if (newSize == null) {
                    newSizeCss = "";
                } else {
                    newSizeCss = newSize;
                }

                this._fixedSide.css(this._cssSizeProp, newSizeCss);
                this._fillSide.css(this._cssPosProp, newSizeCss);
                this.handleBar.css(this._cssPosProp, newSizeCss);

                this._fixedSidePixels = newSize;
                if (!suppressFireResize) {
                    this._fireWindowResize();
                }
            },
            _handleBarDoubleClick: function (e) {
                if (this.expandState) {
                    this.removeExpand();
                }
                else {
                    if (this._options.fixedSide === "left") {
                        this.expand("right");
                    }
                    else {
                        this.expand("left");
                    }
                }

                this._fire("changed", this);
            },
            removeExpand: function (suppressResize) {
                this._element.removeClass("left-expand");
                this._element.removeClass("right-expand");
                this.expandState = null;

                if (!suppressResize) {
                    this._resize(this._fixedSidePixels);
                } else {
                    this._fireWindowResize();
                }
            },
            expand: function (side) {
                var cssProps;

                if (!side) {
                    side = "left";
                }

                if (side == "left") {
                    this._element.removeClass("right-expand");
                    this._element.addClass("left-expand");
                }
                else {
                    this._element.removeClass("left-expand");
                    this._element.addClass("right-expand");
                }


                this.expandState = side;

                this._fixedSide.css(this._cssSizeProp, "");
                this._fillSide.css(this._cssPosProp, "");
                this.handleBar.css(this._cssPosProp, "");

                this._fireWindowResize();
            },
            toggleSplit: function (visible) {

                if (visible) {
                    this._element.removeClass("no-split");

                    this._resize(this._fixedSidePixels);
                }
                else {
                    this._element.addClass("no-split");
                    this._fixedSide.css(this._cssSizeProp, "");
                    this._fillSide.css(this._cssPosProp, "");
                    this.handleBar.css(this._cssPosProp, "");

                    this._fireWindowResize();
                }
            },
            noSplit: function () {
                this.toggleSplit(false);
            },
            split: function () {
                this.toggleSplit(true);
            },
            toggleOrientation: function (vertical, newSize) {
                vertical = Boolean(vertical);

                if (this._options.vertical != vertical) {
                    this._element.removeClass("left-expand");
                    this._element.removeClass("right-expand");
                    this.expandState = null;
                    this._fixedSide.css(this._cssSizeProp, "");
                    this._fillSide.css(this._cssPosProp, "");
                    this.handleBar.css(this._cssPosProp, "");

                    this._options.vertical = vertical;
                    this._element.toggleClass("vertical", vertical);
                    this._element.toggleClass("horizontal", !vertical);
                    this._configureCssProps();

                    this._resize(newSize);
                }
            },
            vertical: function () {
                this.toggleOrientation(true);
            },
            horizontal: function () {
                this.toggleOrientation(false);
            }
        };
    }());



    function CheckboxList(options) {
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "checkbox-list",
            tagName: "ul"
        }, options));
    }

    CheckboxList.extend({
        _typeName: "tfs.checkboxList"
    });

    CheckboxList.inherit(Controls.BaseControl, function () {
        return {
            _checkedItems: null,
            _idMap: null,
            _initializeElement: function () {
                if (!this._options.id) {
                    this._options.id = "cbl_" + Controls.getId();
                }

                this._base();
                this._element.click(delegate(this, this._onCheckClick));
                this._draw();
            },
            _enhance: function (element) {
                this._base(element);
            },
            _draw: function () {
                var that = this;
                this._element.empty();
                this._checkedItems = {};
                this._idMap = {};
                $.each(this._options.items || [], function (i, item) {
                    var text, value, title, checked, $li, $cb, index = i, id = that._options.id + "_cb" + index;

                    if (typeof item === "string") {
                        text = item;
                    }
                    else if (item != null) {
                        text = item.text;
                        value = item.value;
                        title = item.title;
                        checked = item.checked;
                    }

                    text = text || value;

                    if (value == null) {
                        value = text;
                    }

                    if (title == null) {
                        title = text;
                    }

                    if (text != null) {
                        $li = $("<li />").attr("title", title);
                        $cb = $("<input />")
                            .data("value", value)
                            .attr("type", "checkbox")
                            .attr("id", id)
                            .data("index", index)
                            .appendTo($li);

                        if (checked) {
                            $cb.attr("checked", true);
                            that._checkedItems[value] = true;
                        }

                        that._idMap[value] = id;

                        $("<label />").attr("for", id).text(text).appendTo($li);
                        that._element.append($li);
                    }
                });
            },
            enableElement: function (enabled) {
                this._base(enabled);

                if (enabled) {
                    this._element.find("input").removeAttr("disabled");
                }
                else {
                    this._element.find("input").attr("disabled", "disabled");
                }
            },
            setItems: function (items) {
                this._options.items = items;
                this._draw();
            },
            _onCheckClick: function (e) {
                var $target = $(e.target), value;

                if ($target.is("input[type=checkbox]")) {
                    value = $target.data("value");
                    if ($target.attr("checked")) {
                        this._checkedItems[value] = true;
                    }
                    else {
                        delete this._checkedItems[value];
                    }

                    this._fireChange();
                }
            },
            getCheckedValues: function () {
                var result = [], key;

                for (key in this._checkedItems) {
                    result.push(key);
                }

                return result;
            },
            getCheckedItems: function () {
                var that = this;
                return $.map(this._options.items || [], function (item) {
                    var value;

                    if (typeof item === "string") {
                        value = item;
                    }
                    else {
                        value = item.value || item.text;
                    }

                    if (value != null && (value in that._checkedItems)) {
                        return item;
                    }
                });
            },
            setCheckedValues: function (values) {
                var valueMap = {}, that = this, v;

                $.each(values, function (i, v) {
                    valueMap[v] = true;
                });

                $.each(this.getCheckedValues(), function (i, v) {
                    if (v in valueMap) {
                        delete valueMap[v];
                    }
                    else {
                        that._element.find("#" + that._idMap[v]).attr("checked", false);
                        delete that._checkedItems[v];
                    }
                });

                for (v in valueMap) {
                    this._element.find("#" + this._idMap[v]).attr("checked", true);
                    this._checkedItems[v] = true;
                }
            }
        };
    }());


    function FilterControl(options) {
        this.baseConstructor.call(this, $.extend({
            enableRowAddRemove: true,
            enableGrouping: true,
            coreCssClass: "filter-control"
        }, options));
    }

    FilterControl.extend({
        _typeName: "tfs.filterControl"
    });

    FilterControl.extend(TFS.Host.TfsContext.ControlExtensions);

    FilterControl.inherit(Controls.BaseControl, function () {
        return {

            _clauseTable: null,
            _groupHeaderCell: null,
            _filter: null,

            //
            // Abstract methods
            //
            _getDefaultClause: null,
            _updateAndOrControl: null,
            _updateFieldControl: null,
            _updateOperatorControl: null,
            _updateValueControl: null,
            _validateClause: null,
            _handleFieldNameChanged: null,
            _handleOperatorChanged: null,
            _setDirty: null,
            //
            // End Abstract methods
            //

            _createElement: function () {
                this._base();
            },
            setFilter: function (filter) {
                this._filter = filter;
                this._createClauseTable();
            },
            _createClauseTable: function () {
                var that = this;
                if (this._clauseTable) {
                    this._clauseTable.remove();
                    this._clauseTable = null;
                    this._groupHeaderCell = null;
                }

                if (this._filter) {
                    this._clauseTable = $("<table />").addClass("clauses");
                    this._clauseTable.append(this._createHeaderRow());

                    if (!this._filter.clauses || this._filter.clauses.length == 0) {
                        this._filter.clauses = [this._getDefaultClause()];
                    }

                    $.each(this._filter.clauses, function (i, clause) {
                        clause.index = i;
                        that._clauseTable.append(that._createClauseRow(clause));
                    });

                    if (this._options.enableRowAddRemove) {
                        this._clauseTable.append(this._createAddClauseRow());
                    }

                    this._element.append(this._clauseTable);
                }
            },
            _createHeaderRow: function () {
                var that = this, $row = $("<tr/>").addClass("header clause-row"),
					$operatorHeaderCell;

                if (this._options.enableRowAddRemove) {
                    $row.append($("<td/>").addClass("add-remove"));
                }

                if (this._options.enableGrouping) {
                    $row.append();
                    $("<td/>")
						.addClass("grouping disabled")
						.append($("<a/>")
								.attr("href", "#")
								.attr("disabled", "disabled")
								.css("opacity", 0.25)
								.append($("<span/>").addClass("icon"))
								.attr("title", Resources.FilterGroupClauses || "")
								.click(function (e) {
								    if (!$(this).attr("disabled")) {
								        that._groupSelectedClauses();
								    }
								    return false;
								}))
						.appendTo($row);

                    this._groupHeaderCell = $("<td/>").addClass("groups");

                    if (this._filter && this._filter.maxGroupLevel) {
                        this._groupHeaderCell.attr("colspan", this._filter.maxGroupLevel + 1);
                    }

                    $row.append(this._groupHeaderCell);
                }

                if (!this._options.hideLogicalOperator) {
                    $row.append($("<td/>").addClass("logical").text(Resources.FilterControlAndOr));
                }

                $row.append($("<td/>").addClass("field").text(Resources.FilterControlField));
                $operatorHeaderCell = $("<td/>").addClass("operator");
                $row.append($operatorHeaderCell);

                if (!this._options.hideOperatorHeader) {
                    $operatorHeaderCell.text(Resources.FilterControlOperator);
                }

                $row.append($("<td/>").addClass("value").text(Resources.FilterControlValue));

                return $row;
            },

            _getInsertClauseTooltipText: function () {
                return Resources.FilterControlInsertClause;
            },

            _getRemoveClauseTooltipText: function () {
                return Resources.FilterControlRemoveClause;
            },

            _createClauseRow: function (clause) {
                var rowNumber, $row, that = this, groups, group, i, $cell, colSpan, andOrControl, fieldControl, operatorControl, valueControl;

                rowNumber = clause.index + 1;
                $row = $("<tr/>").addClass("clause clause-row").data("clause", clause);

                function getClauseInfo() {
                    return {
                        clause: clause,
                        $row: $row,
                        logicalOperatorControl: andOrControl,
                        fieldNameControl: fieldControl,
                        operatorControl: operatorControl,
                        valueControl: valueControl
                    };
                }

                if (this._options.enableRowAddRemove) {
                    $row.append($("<td/>").addClass("add-remove")
                                    .append($("<a/>").attr("href", "#")
                                            .append($("<span/>").addClass("icon add-icon"))
                                            .attr("title", that._getInsertClauseTooltipText() || "")
                                            .click(function (e) {
                                                return that._addClauseClick(e, getClauseInfo());
                                            }))
                                    .append($("<a/>").attr("href", "#")
                                            .append($("<span/>").addClass("icon icon-delete"))
                                            .attr("title", that._getRemoveClauseTooltipText() || "")
                                            .click(function (e) {
                                                return that._removeClauseClick(e, getClauseInfo());
                                            })));
                }

                if (this._options.enableGrouping) {
                    $row.append($("<td/>").addClass("grouping")
                                    .append($("<input/>").attr("type", "checkbox").data("clauseRow", rowNumber).click(function (e) {
                                        that._updateGroupLink();
                                    })));

                    if (this._filter.groups && this._filter.groups.length) {
                        groups = [];

                        $.each(this._filter.groups, function (i, g) {
                            if (g.start <= rowNumber && g.end >= rowNumber) {
                                g.level = g.level || 0;
                                groups.push(g);
                            }
                        });

                        groups.sort(function (g1, g2) { return g2.level - g1.level; });
                    }

                    if (groups && groups.length) {

                        function appendGroupCell() {
                            if (group) {
                                $cell.addClass("group g-cat-" + ((group.level || 0) % 5));

                                if (group.start === rowNumber) { //row is a group start
                                    $cell.addClass("group-start");

                                    $("<a/>").attr("href", "#")
                                                    .append($("<span/>").addClass("icon icon-tfs-clause-ungroup"))
                                                    .attr("title", Resources.FilterControlUngroupClauses || "")
                                                    .click(function (g) {
                                                        return function (e) {
                                                            return that._ungroupClick(e, $.extend(getClauseInfo(), { group: g }));
                                                        }
                                                    }(group))
                                                    .appendTo($cell);
                                }
                                else {
                                    $("<div/>").addClass("group-placeholder").appendTo($cell);
                                }

                                if (group.end === rowNumber) {
                                    $cell.addClass("group-end");
                                }
                            }

                            if (colSpan > 1) {
                                $cell.attr("colspan", colSpan);
                            }

                            $row.append($cell);
                        }

                        $cell = $("<td/>");
                        colSpan = 0;
                        for (i = this._filter.maxGroupLevel || 0; i >= 0; i--) {
                            if (groups.length > 0) {
                                if ((groups[0].level || 0) == i) {
                                    if (colSpan > 0) {
                                        appendGroupCell();
                                        $cell = $("<td/>");
                                        colSpan = 0;
                                    }

                                    group = groups.shift();
                                }
                            }

                            colSpan++;
                        }

                        if (colSpan > 0) {
                            appendGroupCell();
                        }
                    }
                    else {
                        $row.append($("<td />").addClass("no-group").attr("colspan", (this._filter.maxGroupLevel || 0) + 1));
                    }
                }

                function clauseChanged(change) {
                    that._onClauseChange(change, getClauseInfo());
                }

                if (!this._options.hideLogicalOperator) {
                    $cell = $("<td/>").addClass("logical");
                    if (rowNumber > 1) {
                        andOrControl = Combo.createIn($cell, { mode: "drop", allowEdit: false, change: function () { clauseChanged("logicalOperator"); } });
                        this._updateAndOrControl(andOrControl, clause);
                    }

                    $row.append($cell);
                }

                $cell = $("<td/>").addClass("field");
                fieldControl = Combo.createIn($cell, { mode: "drop", change: function () { clauseChanged("fieldName"); } });
                this._updateFieldControl(fieldControl, clause);
                $row.append($cell);

                $cell = $("<td/>").addClass("operator");
                operatorControl = Combo.createIn($cell, { mode: "drop", sorted: false, source: [], change: function () { clauseChanged("operator"); } });
                this._updateOperatorControl(operatorControl, clause);

                $row.append($cell);

                $cell = $("<td/>").addClass("value");
                valueControl = Combo.createIn($cell, { mode: "text", change: function () { clauseChanged("value"); } });
                this._updateValueControl(valueControl, clause);

                $row.append($cell);

                this._validateClause(getClauseInfo());

                return $row;
            },
            _getAddNewClauseText: function () {
                /// <summary>Gets the string to be displayed in place of "add new clause" hyperlink.</summary>
                return Resources.FilterControlAddNewClause;
            },
            _createAddClauseRow: function () {
                var $row = $("<tr/>").addClass("add-clause clause-row"), $cell, that = this,
                    addNewClauseText = this._getAddNewClauseText();

                function addClauseClickHandler(e) {
                    return that._addClauseClick(e, { $row: $row });
                }

                $cell = $("<td/>").addClass("add-remove")
                            .append($("<span/>").addClass("icon action add-icon").attr("title", addNewClauseText)
                                      .click(addClauseClickHandler))
                            .append($("<a/>").attr("href", "#").addClass("add-row-link")
                                     .append(addNewClauseText)
                                     .click(addClauseClickHandler));

                if (this._options.enableGrouping) {
                    $cell.attr("colspan", 5 + (this._filter.maxGroupLevel || 0));
                }
                else {
                    $cell.attr("colspan", 4);
                }

                $row.append($cell);
                return $row;
            },
            _onClauseChange: function (change, clauseInfo) {
                var oldValue, changed, clause = clauseInfo.clause;

                switch (change) {
                    case "logicalOperator":
                        oldValue = clause.logicalOperator;
                        clause.logicalOperator = clauseInfo.logicalOperatorControl.getText();
                        if (String.localeIgnoreCaseComparer(oldValue, clause.logicalOperator) !== 0) {
                            changed = true;
                        }
                        break;
                    case "fieldName":
                        oldValue = clause.fieldName;
                        clause.fieldName = $.trim(clauseInfo.fieldNameControl.getText());
                        if (String.localeIgnoreCaseComparer(oldValue, clause.fieldName) !== 0) {
                            changed = true;
                            this._handleFieldNameChanged(clauseInfo, oldValue);
                        }
                        break;
                    case "operator":
                        oldValue = clause.operator;
                        clause.operator = $.trim(clauseInfo.operatorControl.getText());
                        if (String.localeIgnoreCaseComparer(oldValue, clause.operator) !== 0) {
                            changed = true;
                            this._handleOperatorChanged(clauseInfo, oldValue);
                        }
                        break;
                    case "value":
                        oldValue = clause.value;
                        clause.value = clauseInfo.valueControl.getText();
                        if (String.localeIgnoreCaseComparer(oldValue, clause.value) !== 0) {
                            changed = true;
                        }
                        break;
                    default:
                }

                if (changed) {
                    this._validateClause(clauseInfo);
                    this._handleFilterModified();
                }
            },

            _addClauseClick: function (e, clauseInfo) {
                var clause, i, l;
                this._filter.clauses = this._filter.clauses || [];
                clause = this._getDefaultClause();

                if (clauseInfo.clause) {
                    clause.index = clauseInfo.clause.index;
                    this._filter.clauses.splice(clauseInfo.clause.index, 0, clause); //insert our clause

                    for (i = clause.index + 1, l = this._filter.clauses.length; i < l; i++) {
                        this._filter.clauses[i].index = i;
                    }

                    this._filter.groups = Utils.updateFilterGroups(this._filter.groups, clause.index + 1, true);
                }
                else {
                    clause.index = this._filter.clauses.length;
                    this._filter.clauses.push(clause);
                }

                this._createClauseTable();

                $(String.format("td.field:eq({0}) input", clause.index + 1), this._element).focus();

                this._handleFilterModified();

                return false;
            },
            _removeClauseClick: function (e, clauseInfo) {
                var index = clauseInfo.clause.index, i, l;
                this._filter.clauses.splice(index, 1); //remove our clause

                l = this._filter.clauses.length;
                for (i = index; i < l; i++) {
                    this._filter.clauses[i].index = i;
                }

                this._filter.groups = Utils.updateFilterGroups(this._filter.groups, index + 1, false);
                this._filter.maxGroupLevel = Utils.updateFilterGroupLevels(this._filter.groups);

                this._createClauseTable();

                this._handleFilterModified();

                return false;
            },
            _updateGroupLink: function () {
                var length = 0, count = 0, prev = false, $headerCell;

                $("td.grouping input", this._element).each(function (i, cb) {
                    if ($(cb).attr("checked")) {
                        length++;
                        if (prev || count === 0) {
                            count++;
                        }

                        prev = true;
                    }
                    else {
                        prev = false;
                    }
                });

                $headerCell = $(".header td.grouping", this._element);
                if (length > 1 && length === count) {
                    $headerCell.removeClass("disabled");
                    $headerCell.find("a").removeAttr("disabled").css("opacity", 1);
                }
                else {
                    $headerCell.addClass("disabled");
                    $headerCell.find("a").attr("disabled", "disabled").css("opacity", 0.25);
                }

                return false;
            },
            _groupSelectedClauses: function () {
                var firstRow = Number.MAX_VALUE, lastRow = 0, hasError, groups = this._filter.groups || [];
                $("td.grouping input:checked", this._element).each(function (i, cb) {
                    var rowNumber = $(cb).data("clauseRow");
                    firstRow = Math.min(firstRow, rowNumber);
                    lastRow = Math.max(lastRow, rowNumber);
                });

                if (firstRow >= lastRow) {
                    alert(Resources.FilterGroupingCannotGroup);
                    return;
                }

                $.each(groups, function (i, g) {
                    if (g.start === firstRow && g.end === lastRow) {
                        alert(Resources.FilterGroupingGroupAlreadyExist);
                        hasError = true;
                        return false;
                    }

                    if ((g.start < firstRow && g.end > firstRow && g.end < lastRow)
                            || (g.start < lastRow && g.end > lastRow && g.start > firstRow)) {
                                alert(Resources.FilterGroupingCannotIntersect);
                                hasError = true;
                                return false;
                            }

                    if (g.end === firstRow || g.start === lastRow) {
                        alert(Resources.FilterGroupingCannotGroup);
                        hasError = true;
                        return false;
                    }
                });

                if (!hasError) {
                    groups.push({ start: firstRow, end: lastRow });
                    this._filter.groups = groups;
                    this._filter.maxGroupLevel = Utils.updateFilterGroupLevels(groups);
                    this._createClauseTable();
                    this._handleFilterModified();
                }
            },
            _ungroupClick: function (e, clauseInfo) {
                var newGroups = [];

                $.each(this._filter.groups, function (i, g) {
                    if (clauseInfo.group.start != g.start || clauseInfo.group.end != g.end) {
                        newGroups.push(g);
                    }
                });

                this._filter.groups = newGroups
                this._filter.maxGroupLevel = Utils.updateFilterGroupLevels(newGroups);

                this._createClauseTable();
                this._handleFilterModified();

                return false;
            },
            _handleFilterModified: function () {
                this._setDirty();
                this._fire("filterModified", this._filter);
            }
        };
    }());

    function StatusIndicator(options) {
        this.baseConstructor.call(this, $.extend({
            coreCssClass: "status-indicator"
        }, options));
    }

    StatusIndicator.extend({
        _typeName: "tfs.statusIndicator"
    });

    StatusIndicator.inherit(Controls.BaseControl, function () {
        return {
            _lastError: null,
            _statusDiv: null,
            _image: null,
            _throttleMinTime: null,

            initialize: function () {
                this._base();

                if (this._options.center && this._options.center === true) {
                    this._element.addClass("center");
                }

                //If no min time specified, then default to 100ms before showing indicator
                this._throttleMinTime = this._options.throttleMinTime || 100;

                //Bind to start, complete and error events if specified
                this._bindEvents();

                this._bind("click", delegate(this, this._onClick));
            },
            _dispose: function () {
                this._clearTimeout();
                this._base();
            },
            _draw: function () {
                var table, tr, td, messageContainer,
                    element = this._element;

                element.empty();
                this._statusDiv = $("<div />").appendTo(element).addClass("status");

                if (!this._options.center) {
                    this._statusDiv.addClass("status-inline");
                    element.addClass("inline");
                    this._image = $("<span />").appendTo(element);
                    messageContainer = element;
                } else {
                    //if we need to center the indicator, we will put it in a 1 cell table
                    table = $("<table />").appendTo(this._statusDiv);
                    tr = $("<tr />").appendTo(table);
                    td = $("<td />").appendTo(tr);
                    this._image = $("<span />").appendTo(td);
                    messageContainer = td;
                }

                //If an image class is specified then use it.  Default to small progress indicator
                this._setImageClass();

                //add a message if one is specified
                if (this._options.message) {
                    $("<span />").appendTo(messageContainer).text(this._options.message);
                }

                this._hide();
            },
            _onClick: function () {
                if (this._lastError) {
                    alert(this._lastError);
                }
            },
            _setImageClass: function () {
                this._image.removeClass();
                if (this._options.imageClass) {
                    this._image.addClass("icon");
                    this._image.addClass(this._options.imageClass);
                } else {
                    this._image.addClass("status-progress");
                }
            },
            _bindEvents: function () {
                var eventTarget = this._options.eventTarget || window;
                if (this._options.statusStartEvent) {
                    this._bind(eventTarget, this._options.statusStartEvent, delegate(this, this._startHandler), true);
                }

                if (this._options.statusCompleteEvent) {
                    this._bind(eventTarget, this._options.statusCompleteEvent, delegate(this, this.complete), true);
                }

                if (this._options.statusErrorEvent) {
                    this._bind(eventTarget, this._options.statusErrorEvent, delegate(this, this.error), true);
                }
            },
            _show: function () {
                this._element.css("display", "");
            },
            _hide: function () {
                this._element.css("display", "none");
            },
            _error: function (e, xhr, settings, exception) {
                this._lastError = exception;
                this._hide();
            },
            _startHandler: function (event, options) {
                this.delayExecute("start", this._throttleMinTime, true, function () {
                    this.start(event, options);
                });
            },
            start: function (event, options) {
                this._clearTimeout();

                //merge in new options
                $.extend(this._options, options);

                if (!this._statusDiv) {
                    this._draw();
                } else {
                    //update the image
                    this._setImageClass();
                }

                this._show();
            },
            complete: function () {
                this._clearTimeout();
                this._hide();
            },
            error: function (exception) {
                this._lastError = exception;
                this._clearTimeout();
                this._hide();
            },
            _clearTimeout: function () {
                this.cancelDelayedFunction("start");
            }
        };
    }());

    function Histogram(options) {
        options.coreCssClass = "histogram";
        this.baseConstructor.call(this, options);
    }

    Histogram.inherit(Controls.BaseControl, function (undefined) {
        return {
            _getBarWidth: function () {
                var barWidth = this._options.barWidth || 5;
                // Ensuring bar width between 2 and 30 px
                barWidth = Math.min(Math.max(barWidth, 2), 30);
                return barWidth;
            },
            _getBarSpacing: function () {
                var barSpacing = this._options.barSpacing || 2;
                // Ensuring bar spacing between 0 and 10 px
                barSpacing = Math.min(Math.max(barSpacing, 0), 10);
                return barSpacing;
            },
            _getBarMaxHeight: function () {
                var barHeight = this._options.barHeight || 35;
                // Ensuring max bar height between 30 and 300 px
                barHeight = Math.min(Math.max(barHeight, 10), 300);
                return barHeight;
            },
            _getBarCount: function () {
                var barCount = this._options.barCount || 9;
                // Ensuring bar count between 5 and 25
                barCount = Math.min(Math.max(barCount, 5), 100);
                return barCount;
            },

            initialize: function () {
                var options = this._options;

                this._decorate();

                if ($.isArray(options.items)) {
                    // If any items specified initially, load it
                    this._load(options.items);
                }
                else if ($.isFunction(options.items)) {
                    // If a loader function is specified, call it to get the items
                    this._load(options.items.call(this));
                }
                else if (options.defaultBars !== false) {
                    this._renderDefaultBars();
                }
            },

            _load: function (items) {
                this._renderBars(items);
            },

            refresh: function (items) {
                this._load(items);
            },

            _decorate: function () {
                // Setting width and height
                this._element.width(this._getBarCount() * (this._getBarSpacing() + this._getBarWidth()));
                this._element.height(this._getBarMaxHeight());
            },

            _clearBars: function () {
                this._element.find("div.bar").remove();
            },

            _renderDefaultBars: function () {
                this._clearBars();

                for (var i = 0; i < this._getBarCount() ; i++) {
                    this._element.append(this._createBar(i));
                }
            },

            _renderBars: function (items) {
                var i, l, barCount;

                assert($.isArray(items), "Array of items expected.");

                this._clearBars();

                barCount = this._getBarCount();
                l = barCount - items.length;

                // If there is not enough items, rendering the default items first
                for (i = 0; i < l; i++) {
                    this._element.append(this._createBar(i, { state: "default", value: 5 }));
                }

                // Rendering actual items
                while (i < barCount) {
                    this._element.append(this._createBar(i, items[i - l]));
                    i += 1;
                }
            },

            _createBar: function (index, item) {
                var left,
                    barElement,
                    barWidth,
                    state,
                    height,
                    hoverState = this._options.hoverState,
                    selectedState = this._options.selectedState,
                    allowInteraction = this._options.allowInteraction !== false;

                barWidth = this._getBarWidth();
                barSpacing = this._getBarSpacing();

                state = item && item.state;
                state = state || "default";

                // Calculating positioning related quantities
                left = index * (barWidth + this._getBarSpacing());

                height = item && item.value;
                height = typeof (height) === "number" ? (Math.max(Math.min(height, 100), 0)) : (height || (index % 2 ? 50 : 100));

                // Ensuring the item to be visible
                height = Math.max(height, Math.ceil(100 / this._getBarMaxHeight()));

                barElement = $("<div />").addClass("bar").appendTo(this._element);
                barElement.addClass(state);

                // Positioning bar element
                barElement.css("left", left);
                barElement.css("width", barWidth);
                barElement.css("height", height + "%");
                barElement.css("bottom", 0);

                if (item) {
                    if (selectedState && (item.selected || !allowInteraction)) {
                        // If a selected state exists and item is selected, setting the selected class
                        barElement.addClass(selectedState);
                    }
                    if (allowInteraction && hoverState && !item.selected) {
                        // If a hover state exists and the item is not selected, attaching to hover events
                        barElement.hover(function () {
                            $(this).addClass(hoverState);
                        },
                           function () {
                               $(this).removeClass(hoverState);
                           });
                    }
                    if (allowInteraction && $.isFunction(item.action)) {
                        // If the item has an action, attaching to click event
                        barElement.data("action", { action: item.action, args: item.actionArgs });
                        barElement.click(function () {
                            var action = $(this).data("action");
                            action.action(action.args);
                            return false;
                        });
                    }

                    if (item.title) {
                        // If the item has a title, adding it
                        barElement.attr("title", item.title);
                    }
                }
            }
        };
    }());

    function TileZone(options) {
        /// <summary>Represents a tile zone on the client side</summary>

        options = $.extend({
            previewOpacity: 0.7,
            previewScale: 25
        }, options);

        this.baseConstructor.call(this, options);
    }

    TileZone.inherit(Controls.BaseControl, {
        _tileZoneId: null,
        _tileCount: null,
        _keyboardReorderTile: null,
        _keyboardReorderTileIndex: null,

        _attachEvents: function () {
            /// <summary>Attaches keyup and keydown for the TileZone, and blur for the window</summary>
            this._bind(this._element, "keyup", delegate(this, this._onKeyUp));
            this._bind(this._element, "keydown", delegate(this, this._onKeyDown));
            this._bind(window, "blur", delegate(this, this._onBlur));
        },

        _clearReorderTileState: function () {
            /// <summary>Clears the keyboard reorder tile state</summary>
            Diag.assertIsNotNull(this._keyboardReorderTile, "Unexpected, there is no reorder tile state.");

            this._keyboardReorderTile.css("opacity", "1.0");
            this._keyboardReorderTile = null;
            this._keyboardReorderTileIndex = null;
        },

        _getTileIndex: function ($tile) {
            /// <summary>Gets the current index for the tile</summary>
            /// <param name="$tile" type="jQuery">The tile wrapped in a jQuery object</param>
            /// <returns type="Number" />
            Diag.assertParamIsJQueryObject($tile, "$tile");

            var $tiles = $(".tile", this._element);
            return $tiles.index($tile);
        },

        _isKeyboardMovementValid: function (index, keyCode) {
            /// <summary>Determines whether this is a valid keyboard movement for reordering purposes</summary>
            /// <param name="index" type="Number">The index of the tile to move</param>
            /// <param name="keyCode" type="Number">The key code pressed</param>
            /// <returns type="Boolean" />
            Diag.assertParamIsNumber(index, "index");
            Diag.assertParamIsNumber(keyCode, "keyCode");

            var validKey = keyCode === TFS.UI.KeyCode.LEFT || keyCode === TFS.UI.KeyCode.RIGHT,
                firstTileMovingLeft = index === 0 && keyCode === TFS.UI.KeyCode.LEFT,
                lastTileMovingRight = index === this._tileCount - 1 && keyCode === TFS.UI.KeyCode.RIGHT;

            return validKey && !firstTileMovingLeft && !lastTileMovingRight;
        },

        _isKeyboardReorderGestureActive: function (e) {
            /// <summary>Determines whether the given event is the gesture to reorder with the keyboard</summary>
            /// <param name="e" type="Object">Event args</param>
            /// <returns type="Boolean" />

            return e.shiftKey;
        },

        _onBlur: function () {
            /// <summary>Responsible for restoring the transparent tile to opaque should the window lose
            // focus in the midst of a keyboard reorder, and saving any in-progress reorder</summary>

            this._setTileOrderFromKeyboard();
        },

        _onKeyUp: function (e) {
            Diag.assertParamIsObject(e, "e");

            // If the keyboard reorder gesture key is released.
            if (!this._isKeyboardReorderGestureActive(e)) {

                // Save changes.
                this._setTileOrderFromKeyboard();
            }
            else if (this._isKeyboardReorderGestureActive(e) && e.keyCode === TFS.UI.KeyCode.TAB) {

                    // Since keyboard reorder gesture is still active, select the new tile for reorder.
                this._reorderTileFromKeyboard(e);
            }
        },

        _onKeyDown: function (e) {
            Diag.assertParamIsObject(e, "e");

            if (this._isKeyboardReorderGestureActive(e)) {
                this._reorderTileFromKeyboard(e);

                if (e.keyCode === TFS.UI.KeyCode.TAB) {
                    // Since keyboard reorder gesture is still active when Tab is pressed,
                    // commit any reorder changes, and allow navigation to go to previous element.
                    this._setTileOrderFromKeyboard();
                }
            }
        },

        _reorderTileFromKeyboard: function (e) {
            /// <summary>Responsible for reordering the tile by updating the DOM if the keyboard event is valid</summary>

            // Get the current tile that is being reordered, and it's current index.
            var $tile = $(e.target).closest(".tile"),
                sourceIndex = this._getTileIndex($tile);

            // If this is the first keyboard reorder key sequence then initialize the state that tracks tile being reordered.
            if (this._keyboardReorderTile === null) {
                this._setReorderTileState($tile, sourceIndex);
            }

            if (this._isKeyboardMovementValid(sourceIndex, e.keyCode)) {
                // By moving the sibling and not the selected tile the focus remains with the selected tile.
                if (e.keyCode === TFS.UI.KeyCode.LEFT) {
                    // Insert the previous tile after our selected tile.
                    var $prevTile = this._keyboardReorderTile.prev();
                    $prevTile.insertAfter(this._keyboardReorderTile);
                }
                else if (e.keyCode === TFS.UI.KeyCode.RIGHT) {
                        // Insert the next tile before our selected tile.
                    var $nextTile = this._keyboardReorderTile.next();
                    $nextTile.insertBefore(this._keyboardReorderTile);
                }
            }
        },

        _setReorderTileState: function ($tile, index) {
            /// <summary>Sets the keyboard reorder tile state</summary>
            /// <param name="$tile" type="jQuery">The tile wrapped in a jQuery object</param>
            /// <param name="index" type="Number">The index id of the tile</param>
            Diag.assertParamIsJQueryObject($tile, "$tile");
            Diag.assertParamIsNumber(index, "index");

            this._keyboardReorderTile = $tile;
            this._keyboardReorderTile.css("opacity", this._options.previewOpacity);
            this._keyboardReorderTileIndex = index;
        },

        _setTileOrderFromKeyboard: function () {
            /// <summary>Calls setTileOrder if the index of the reordered tile differs from its initial index
            /// and clears the reorder tile state</summary>

            if (this._keyboardReorderTile !== null) {
                var index = this._getTileIndex(this._keyboardReorderTile);
                if (index !== this._keyboardReorderTileIndex) {
                    // If the new index is not the same as the starting index then save the changes.
                    this.setTileOrder();
                }

                // Clear the keyboard state.
                this._clearReorderTileState();
            }
        },

        initialize: function () {
            /// <summary>Initializes the tile zone, making it sortable</summary>
            this._base();

            var that = this;

            this._element.sortable({
                opacity: this._options.previewOpacity,
                cursor: "default",
                cursorAt: { left: 75, top: 75 },
                revert: false,
                helper: "clone",
                containment: "parent",
                tolerance: "pointer",
                start: function (event, ui) {
                    var preview = $(ui.helper);
                    preview.height(preview.height() + that._options.previewScale);
                    preview.width(preview.width() + that._options.previewScale);
                },
                update: function (event, ui) {
                    that.setTileOrder.call(that);
                }
            });

            this._element.disableSelection();

            this._tileZoneId = this._element.data("tile-zone-id");
            this._tileCount = $(".tile", this._element).length;

            this._attachEvents();
        },

        setTileOrder: function () {
            /// <summary>Posts the current tile order to the server, this is a fire and forget operation</summary>

            var $tiles = $(".tile", this._element),
                tileList = new Array();

            // store the ordered tile ids in an array
            $.map($tiles, function (element, index) {
                tileList[index] = $(element).data("tile-id");
            });

            // since this a fire and forget action, we will not handle the error/success callback
            Ajax.postMSJSON(
                TFS.Host.TfsContext.getDefault().getActionUrl("SetTileOrder", "tilezone", { area: "api" }),
                {
                    tileZoneId: this._tileZoneId,
                    data: Utils.stringifyMSJSON(tileList)
                }
            );
        }
    });

    // Register the enhancement for the Team Favorites tile zone only.
    TileZone.registerEnhancement(".tile-zone-team-favorites");

    function Tile(options) {
        /// <summary>Represents a tile object on the client side. Derives from AjaxPanel, which makes necessary
        /// lifting about loading the content using url in the options.</summary>
        this.baseConstructor.call(this, options);
    }

    Tile.inherit(AjaxPanel, function () {
        return {
            initialize: function () {

                // Skipping progress indicator temporarily
                // Parts might have a custom loading indicator
                this._options.showStatusIndicator = false;

                this._base();

                if (!this._options.url) {
                    // If no content URL is specified for this tile, we might still
                    // want to make this tile to be actionable
                    this.tryMakingActionable();
                }
            },

            tryMakingActionable: function () {
                // <summary>Checks any action URL exists or not. If exists, this
                // method wraps the inner content with an &lt;a&gt; element.</summary>

                var actionUrl = this._options.actionUrl,
                    actionTarget = this._options.actionTarget,
                    href;

                if (actionUrl) {
                    href = $("<a />").attr("href", actionUrl);

                    if (actionTarget) {
                        href.attr("target", actiontarget);
                    }

                    this._element.wrapInner(href);
                }
            },

            beginLoad: function (url, params, callback, errorcallback) {
               
                if (this._options.name) {
                    Diag.logTracePoint("Tile." + this._options.name + ".beginLoad.Start")
                }

                this._base(url, params, callback, errorcallback);
            },

            onLoadCompleted: function (content) {
                this._base(content);
                this.tryMakingActionable();

                if (this._options.name) {
                    Diag.logTracePoint("Tile." + this._options.name + ".beginLoad.Complete")
                }
            },

            showError: function (error) {
                var element = this._element,
                    errorMessage = getErrorMessage(error);

                // Decorating tile to display error
                element.empty().addClass("error").attr("title", errorMessage);

                if (this._options.context && this._options.context.name) {
                    $("<div />").text(this._options.context.name).appendTo(element);
                }

                $("<p />").text(errorMessage || "").appendTo(element);
                this.tryMakingActionable();
            }
        };
    }());

    Tile.registerEnhancement(".tile");

    function LongRunningOperation(container, options) {
        /// <summary>Creates a new long running operation, showing a blocking indicator in a cancellable means overtop the specified container until the operation has completed.</summary>
        /// <param name="container" type="Object">A DOM object that contains the control on the page in which to overlay the progress indicator.</params>
        /// <param name="options" type="Object">A collection of configuration name/value pairs.  The following are supported:
        ///     Name                  Type        Value
        ///     cancellable           boolean     Boolean value indicating whether the operation may be cancelled while it is running.
        ///</param>

        Diag.assertParamIsObject(container, "container");

        this._$rootElement = $(container);
        this._options = options || {};

        this._initialize();
    }

    LongRunningOperation.extend({});

    LongRunningOperation.prototype = {
        _cancelable: null,
        _options: null,
        _$rootElement: null,
        _waitControl: null,
        _state: null,

        _initialize: function () {
            /// <summary>Initializes the long running operation.</summary>

            var hostConfig = TFS.Host.TfsContext.getDefault().configuration;

            this._state = {
                message: this._options && this._options.message ? this._options.message : Resources.DefaultWaitMessage,
                target: this._$rootElement[0]._element,
                cancellable: this._options && this._options.cancellable
            };

            this._waitControl = new WaitControl(this._state);
        },

        beginOperation: function (operationCallback) {
            /// <summary>Begins the long running operation, invoking the specified operationCallback when necessary.</summary>
            /// <param name="operationCallback" type="function">An operation that may take a long time to complete.</param>

            var cancelable = new Core.Cancelable(this);

            this._cancelable = cancelable;

            Diag.assertParamIsFunction(operationCallback, "operationCallback");

            if ($.isFunction(this._options.beginOperationCallback)) {
                this._options.beginOperationCallback();
            }

            this._waitControl.startWait(cancelable);

            // Wait to make sure that the spinner has time to download if it
            // hasn't been downloaded yet.
            Core.delay(this, 100, function () {
                operationCallback(cancelable);
            });
        },

        endOperation: function () {
            /// <summary>Signals the completion of a long running operation.</summary>
            this._waitControl.endWait();

            if ($.isFunction(this._options.endOperationCallback)) {
                this._options.endOperationCallback();
            }
        },

        isCancelled: function () {
            /// <summary>Gets a boolean value indicating whether the current operation has a pending cancel request.</summary>

            return this._cancelled;
        },

        cancelOperation: function () {
            /// <summary>Cancels the current operation.</summary>

            if (this._options && this._options.cancellable) {
                this._cancelled = true;
            }

            this.endOperation();
        }
    };

    function Clipboard() {
    }

    Clipboard.extend({
        FORMAT_TEXT: "text",

        copyToClipboard: function (data, options) {
            /// <summary>Copies the specified data to the clipboard in the TEXT format using a progressively degrading experience.</summary>
            /// <param name="data" type="String">The data to copy.</param>

            if (typeof data === "string") {
                if (Clipboard.supportsNativeCopy()) {
                    Clipboard._nativeCopy(data, options);
                }
                else {
                    Clipboard._copyUsingNewWindow(data, options);
                }
            }
        },

        supportsNativeCopy: function () {
            /// <summary>Gets a boolean value indicating whether the current browser supports native clipboard access.</summary>
            return window.clipboardData;
        },

        _nativeCopy: function (data, options) {
            /// <summary>Copies the specified data in the specified format to the clipboard using native clipboard support based on the W3C HTML5 clipboard interaction specification (http://www.w3.org/TR/clipboard-apis).</summary>
            /// <param name="data" type="String">The data to copy.</param>
            /// <remarks>Currently, IE is the only browser that supports the spec.</remarks>

            if (data) {
                window.clipboardData.setData(Clipboard.FORMAT_TEXT, data);
            }
        },

        _copyUsingNewWindow: function (data, options) {
            /// <summary>To support non-IE browser copy, opens a new popup window and writes the table to the window allowing the user to copy manually.</summary>
            /// <param name="data" type="String">The data to place on the clipboard (via a popup window).</param>

            options = {
                data: data
            };

            CopyContentDialog.show(options);
        }
    });


    function CopyContentDialog(options) {

        options = $.extend({
            width: 600,
            minWidth: 450,
            height: 450,
            minHeight: 450,
            attachResize: true,
            defaultButton: "close",
            buttons: [{
                id: "close",
                text: Resources.CloseButtonLabelText,
                click: delegate(this, function () { this.close(); })
            }]
        }, options);

        this.baseConstructor.call(this, options);
    }

    CopyContentDialog.extend({ _typeName: "CopyContentDialog" });

    CopyContentDialog.inherit(ModalDialog, function (undefined) {

        return {
            _$textArea: null,

            initialize: function () {
                /// <summary>Initializes the dialog.</summary>
                this._base();
                this.setTitle(Resources.CopyContentDialogTitle);

                this._decorate();
            },

            _decorate: function () {
                /// <summary>Initializes the dialog UI.</summary>
                var $element = this._element,
                    $dialogLabel = $("<div>").text(Resources.CopyContentHelpText),
                    $textPanel = $("<div>"),
                    $layoutPanel = $("<div>")
                        .append($dialogLabel)
                        .append($("<p>"))
                        .append($textPanel);


                this._initializeTextPanel($textPanel);

                $element.append($layoutPanel);

                Core.delay(this, 100, function () {
                    this._$textArea.focus();
                    this._$textArea.select();
                });
            },

            _initializeTextPanel: function ($container) {
                /// <summary>Initializes the text area panel</summary>
                /// <param name="$container" type="JQuery Object">The text area panel container.</param>
                TFS.Diag.assertParamIsObject($container, "$container");

                var $textAreaElement = $("<div>");

                this._$textArea = $("<textarea/>").addClass("copy-content-textarea")
                                                       .text(this._options.data)
                                                       .appendTo($textAreaElement);

                this._$textArea[0].oncopy = delegate(this, function () {
                    Core.delay(this, 0, function () {
                        this.close();
                    });
                });

                $container.append($textAreaElement);
            }
        };
    }());

    /////////////////////////////////////////////////
    /// WaitControl
    /////////////////////////////////////////////////
    function WaitControl(options) {
        /// <summary>Constructs a WaitControl object.</summary>
        /// <param name="options" type="object">The options to initialize the control. It has the following properties:
        ///   {
        ///       image: hostConfig.getResourcesFile('big-progress.gif'),   // optional
        ///       message: "Please wait...",                                // optional
        ///       target: $('.feedbackrequest-form-container')              // optional
        ///       cancellable: true                                         // optional
        ///       cancelCallback: function() { // do something }            // optional and only effective when cancellable is true
        ///   }
        /// </param>
        /// <returns type="object">A WaitControl object.</returns>
        Diag.assertParamIsObject(options, "options");

        this._options = $.extend({
            image: TFS.Host.TfsContext.getDefault().configuration.getResourcesFile('big-progress.gif')
        }, options);
    }

    WaitControl.extend({
        _instanceIdSeed: 1,
        DefaultShowDelay: 250,
        MinLifeTime: 100,
        WaitingState: { Unstarted: 0, Waiting: 1, Ending: 2, Ended: 3, Cancelling: 4, Cancelled: 5 }
    });

    WaitControl.prototype = {
        _options: null,
        _waitContext: null,
        _waitingState: WaitControl.WaitingState.Unstarted,
        _keyDownEventHandler: null,

        startWait: function (cancelable) {
            /// <summary>Starts waiting.</summary>
            /// <param name="cancelable" type="Object">[optional] A TFS.Core.Cancelable object for additional cancel state signaling.</param>
            if (this._canStartWait()) {

                this._waitContext = {

                    // Assign a unique instance Id for resize event binding
                    instanceId: "waitControl_" + WaitControl._instanceIdSeed++,

                    cancelable: cancelable,

                    // The wait options
                    options: {
                        wait: this._options
                    }
                };

                this._startWait();
            }
        },

        endWait: function () {
            /// <summary>Ends waiting.</summary>
            if (this._canEndWait()) {
                this._waitingState = WaitControl.WaitingState.Ending;
                this._tryEndWait();
            }
        },

        cancelWait: function () {
            /// <summary>Cancels waiting.</summary>
            if (this._canCancelWait()) {
                this._waitingState = WaitControl.WaitingState.Cancelling;
                this._tryCancelWait();
            }
        },

        _canStartWait: function () {
            /// <summary>Determines if the current waiting session can be started.</summary>
            return this._waitingState === WaitControl.WaitingState.Unstarted;
        },

        _canEndWait: function () {
            /// <summary>Determines if the current waiting session can be ended.</summary>
            return this._waitContext &&
                (this._waitingState === WaitControl.WaitingState.Waiting ||
                this._waitingState === WaitControl.WaitingState.Ending);
        },

        _canCancelWait: function () {
            /// <summary>Determines if the current waiting session can be cancelled.</summary>
            return this._waitContext &&
                this._waitContext.options.wait.cancellable &&
                (this._waitingState === WaitControl.WaitingState.Waiting ||
                this._waitingState === WaitControl.WaitingState.Cancelling);
        },

        _startWait: function () {
            /// <summary>Starts the waiting.</summary>
            var that = this,
                wait = this._waitContext.options.wait;

            this._waitingState = WaitControl.WaitingState.Waiting;

            // Make sure we have a showDelay set
            if (!wait.showDelay) {
                wait.showDelay = WaitControl.DefaultShowDelay;
            }

            // If no target was specified, use the entire window as the wait target
            if (!wait.target) {
                wait.entireWindow = true;
                wait.target = $('body');
                wait.extraStyles = "height:100%; width:100%;";
            }

            // Hide any existing waits on children
            wait.target.children('.wait-element').hide();
            wait.target.children('.wait-box').hide();

            // If a showDelay is specified, call showWait after showDelay time
            if (wait.showDelay !== 0) {
                this._waitContext.showTimer = Core.delay(this, wait.showDelay, this._showWait);
            }
            else {
                this._showWait();
            }
        },

        _tryEndWait: function () {
            /// <summary>Ends the waiting.</summary>
            Diag.assertIsObject(this._waitContext, "this._waitContext is not an object");
            var wait = this._waitContext.options.wait;

            // If wait is marked to be ended and the timer's gone off, end the wait.
            if (this._waitingState === WaitControl.WaitingState.Ending && !wait.minLifeSpanBlocking) {
                this._waitingState = WaitControl.WaitingState.Ended;
                this._reset();
            }
        },

        _tryCancelWait: function () {
            /// <summary>Cancels the waiting.</summary>
            Diag.assertIsObject(this._waitContext, "this._waitContext is not an object");
            var wait = this._waitContext.options.wait,
                cancelCallback = wait.cancelCallback,
                cancelable = this._waitContext.cancelable;

            // If wait is marked to be cancelled and the timer's gone off, cancel the wait.
            if (this._waitingState === WaitControl.WaitingState.Cancelling && !wait.minLifeSpanBlocking) {

                this._waitingState = WaitControl.WaitingState.Cancelled;

                this._reset();

                if (cancelCallback) {
                    cancelCallback();
                }

                if (cancelable && cancelable.cancel && $.isFunction(cancelable.cancel)) {
                    cancelable.cancel();
                }
            }
        },

        _reset: function () {
            /// <summary>Sets this wait control.</summary>
            this._unbindKeydownEvent();

            this._removeWaitElement();

            this._removeShowTimer();

            this._waitContext = null;
            this._waitingState = WaitControl.WaitingState.Unstarted;
        },

        _showWait: function () {
            /// <summary>Shows the wait control.</summary>
            Diag.assertIsObject(this._waitContext, "this._waitContext is not an object");
            var msgContent = '',
                wait = this._waitContext.options.wait,
                extraStyles = wait.extraStyles || '',
                that = this,
                waitMessage;

            // Determine what content is required for wait message
            if (wait.image) {
                msgContent += String.format('<img class="wait-image" src="{0}" />', wait.image);
            }

            waitMessage = this._getWaitMessage(wait);
            if (waitMessage) {
                msgContent += String.format('<div class="wait-message">{0}</div>', waitMessage);
            }

            // If the caller wants animation and or a message we need to configure it
            if (msgContent.length > 0) {
                wait.target.prepend(String.format('<div class="wait-box">{0}</div>', msgContent));
                wait.msgElement = wait.target.children('.wait-box').first();
            }

            // Set the custom background color if one was specified
            if (wait.backgroundcolor) {
                extraStyles += String.format('background-color:{0};', wait.backgroundcolor);
            }

            // Build the waitElement and make sure it is sized properly
            wait.target.prepend(String.format('<div class="wait-element" style="{0}"></div>', extraStyles));
            wait.element = wait.target.children('.wait-element').first();

            // Make sure the wait element is laid out properly
            this._resizeWait();

            // Attach to the window resize event to update our position
            $(window).bind(this._getResizeEventId(this._waitContext.instanceId), delegate(this, this._resizeWait));

            if (wait.cancellable) {
                Diag.assertIsStringNotEmpty(wait.cancelLinkId, "wait.cancelLinkId is empty");
                this._bindKeydownEvent(wait.cancelLinkId);
                $("#" + wait.cancelLinkId).click(delegate(this, this._handleCancelEvent));
                $("#" + wait.cancelLinkId).focus();
            }

            // Make sure we have at least a minimal lifetime if it wasn't explicitly set
            if (!wait.minLifetime) {
                wait.minLifetime = WaitControl.MinLifeTime;
            }

            // Don't allow the session to expire until the minLifeSpan timeout. This prevents flashing.
            wait.minLifeSpanBlocking = true;
            Core.delay(this, wait.minLifetime, function () {

                // Clear the minLifeSpanTimeout timer on the object
                wait.minLifeSpanBlocking = false;

                if (that._waitingState === WaitControl.WaitingState.Ending) {
                    that._tryEndWait();
                }
                else if (that._waitingState === WaitControl.WaitingState.Cancelling) {
                    that._tryCancelWait();
                }
            });
        },

        _resizeWait: function () {
            /// <summary>Resizes the waiting control.</summary>
            Diag.assertIsObject(this._waitContext, "this._waitContext is not an object");
            var wait = this._waitContext.options.wait,
                waitElementMarginTop, waitElementMarginLeft;

            Diag.assertIsObject(wait.element, "wait.element is not an object");

            // Update the size and position of the wait element to cover the target properly.
            if (!wait.entireWindow) {
                wait.element.css('margin-top', -parseInt(wait.target.css('padding-top'), 10));
                wait.element.css('margin-left', -parseInt(wait.target.css('padding-left'), 10));
                wait.element.height(wait.target.outerHeight());
                wait.element.width(wait.target.outerWidth());
            }

            // Update the position of the message element if one is shown
            if (wait.msgElement) {
                waitElementMarginTop = parseInt(wait.element.css('margin-top'), 10);
                waitElementMarginLeft = parseInt(wait.element.css('margin-left'), 10);

                // Note that the following normalization is only required for IE8, where
                //   calling css on the wait element returns NaN.
                if (!waitElementMarginTop || isNaN(waitElementMarginTop)) {
                    waitElementMarginTop = 0;
                }

                if (!waitElementMarginLeft || isNaN(waitElementMarginLeft)) {
                    waitElementMarginLeft = 0;
                }

                wait.msgElement.css('top', wait.element.position().top +
                (wait.element.height() - wait.msgElement.height()) / 2 +
                waitElementMarginTop);
                wait.msgElement.css('left', wait.element.position().left +
                (wait.element.width() - wait.msgElement.width()) / 2 +
                waitElementMarginLeft);
            }
        },

        _onKeyDown: function (e) {
            /// <summary>Handles the keydown event.</summary>
            /// <param name="e" type="Object">The keydown event.</param>
            var keyCode = TFS.UI.KeyCode;

            switch (e.keyCode) {
                case keyCode.ESCAPE:
                    this._handleCancelEvent();
                    return false;

                default:
                    return;
            }
        },

        _handleCancelEvent: function () {
            /// <summary>Handles the events to cancel wait.</summary>
            this._unbindKeydownEvent();
            this.cancelWait();

            // This handler is registered to the Esc hyperlink click event.
            // Return false to prevent the handled event from bubbling, so that
            // the default hyperlink click behavior, i.e. navigation, is suppressed.
            return false;
        },

        _bindKeydownEvent: function (cancelLinkId) {
            /// <summary>Binds the keydown event</summary>
            /// <param name="cancelLinkId" type="String">The id of the cancel hyperlink.</param>
            Diag.assertParamIsString(cancelLinkId, "cancelLinkId");
            if (!this._keyDownEventHandler) {
                this._keyDownEventHandler = delegate(this, this._onKeyDown);
                //$(document).keydown(this._keyDownEventHandler);
                $("#" + cancelLinkId).keydown(this._keyDownEventHandler);
            }
        },

        _unbindKeydownEvent: function () {
            /// <summary>Unbinds the keydown event</summary>
            var wait;

            if (this._waitContext) {
                wait = this._waitContext.options.wait;

                if (this._keyDownEventHandler && wait.cancelLinkId) {
                    $("#" + wait.cancelLinkId).unbind("keydown", this._keyDownEventHandler);
                    this._keyDownEventHandler = null;
                }
            }
        },

        _removeWaitElement: function () {
            /// <summary>Removes the wait element.</summary>
            var wait;

            function removeWaitElement() {
                /// <summary>Removes the wait element</summary>
                if (wait.element) {
                    wait.element.remove();
                    wait.element = null;
                }
            }

            function removeMessageElement() {
                /// <summary>Removes the wait message element</summary>
                if (wait.msgElement) {
                    wait.msgElement.remove();
                    wait.msgElement = null;
                }
            }

            if (this._waitContext) {

                wait = this._waitContext.options.wait;

                if (wait.element) {
                    if (wait.fade !== false) {
                        wait.element.fadeOut('fast', removeWaitElement);
                        wait.msgElement.fadeOut('fast', removeMessageElement);
                    }
                    else {
                        removeWaitElement();
                        removeMessageElement();
                    }

                    // Remove the resize binder for moving the wait element
                    $(window).unbind(this._getResizeEventId(this._waitContext.instanceId));
                }
            }
        },

        _removeShowTimer: function () {
            /// <summary>Removes the timers used by this controls.</summary>

            if (this._waitContext && this._waitContext.showTimer) {
                this._waitContext.showTimer.cancel();
                delete this._waitContext.showTimer;
            }
        },

        _getResizeEventId: function (instanceId) {
            /// <summary>Gets the unique resize event id for the wait control.</summary>
            /// <returns type="String">The resize event id.</returns>
            Diag.assertParamIsString(instanceId, "instanceId");
            return 'resize.' + instanceId;
        },

        _getWaitMessage: function (wait) {
            /// <summary>Gets the text message to show in the wait control.</summary>
            /// <param name="wait" type="Object">The wait options.</param>
            Diag.assertParamIsObject(wait, "wait is not an object");
            var message = '',
                cancelAdvice = '';

            if (wait.cancellable) {
                wait.cancelLinkId = "cancelLink_" + this._waitContext.instanceId;
                cancelAdvice = String.format(Resources.CancelWaitAdvice, String.format('<a href="" class="wait-link" tabindex="0" id="{0}">ESC</a>', wait.cancelLinkId));
            }

            if (wait.message) {
                message = (wait.cancellable ?
                        String.format('{0} ({1})', wait.message, cancelAdvice) : wait.message);
            }
            else if (wait.cancellable) {
                message = cancelAdvice;
            }

            return message;
        }
    };

    var MessageAreaType = {
        None: 0,
        Info: 1,
        Warning: 2,
        Error: 3
    };

    //// MessageAreaControl
    function MessageAreaControl(options) {
        this.baseConstructor.call(this, $.extend({
            closeable: true,
            expanded: false
        }, options));
    }

    MessageAreaControl.extend({
        EVENT_DISPLAY_COMPLETE: "event-display-complete"
    });

    MessageAreaControl.inherit(Controls.BaseControl, {
        _errorHeader: null,     // the element showing main message
        _errorContent: null,    // the element showing message details
        _messageType: null,

        initialize: function () {
            var errorHeaderDiv, that = this;

            this._element.addClass('message-area-control');

            errorHeaderDiv = $(domElem('div'))
                .appendTo(this._element)
                .addClass('message-header');
            this._errorHeader = $(domElem('span')).appendTo(errorHeaderDiv);

            this._showErrorLink = $(domElem('a')).appendTo(errorHeaderDiv)
                .addClass('linkAction show-details-action')
                .attr('href', '#')
                .click(function () {
                    that._toggle();
                    return false;
                });

            $(domElem('div'))
                .appendTo(this._element)
                .addClass('close-action icon icon-cancel')
                .click(delegate(this, this.clear)).accessible();

            this._errorContent = $(domElem('div'))
                .appendTo(this._element)
                .addClass('error-content');

            this._element.hide();

            this._messageType = MessageAreaType.None;

            if (this._options.message) {
                this.setMessage(this._options.message);
            }
        },

        setMessage: function (message, messageType) {
            /// <summary>Set the message</summary>
            /// <param name="message" type="Object">Message string or
            ///     options = {
            ///         messageType: <MessageAreaType>,
            ///         header: <String>,
            ///         content: <html String>,
            ///         click: <function>
            ///     }
            /// </param>
            /// <param name="messageType" type="MessageAreaType">Type of message</param>
            if (typeof (message) === 'string') {
                message = {
                    header: message
                };
            }
            Diag.assertParamIsObject(message, 'message');

            if (!message.type) {
                message.type = messageType || MessageAreaType.Error;
            }

            this._setDisplayMessage(message);
        },

        setError: function (message, clickCallback) {
            /// <summary>Set the error message</summary>
            /// <param name="message" type="Object">Message string or
            ///     options = {
            ///         messageType: <MessageAreaType>,
            ///         header: <String>,
            ///         content: <html String>,
            ///         click: <function>
            ///     }
            /// </param>
            /// <param name="clickCallback" type="function">Click callback function</param>

            if (typeof (message) === 'string') {
                message = {
                    header: message
                };
            }
            Diag.assertParamIsObject(message, 'message');

            // Set type to Error
            message.type = MessageAreaType.Error;

            if (!message.click && $.isFunction(clickCallback)) {
                message.click = clickCallback;
            }

            this.setMessage(message);
        },

        _setDisplayMessage: function (message) {
            /// <summary>Set the display message</summary>
            /// <param name="message" type="Object">
            ///     options = {
            ///         messageType: <MessageAreaType>,
            ///         header: <String>,
            ///         content: <html String>,
            ///         click: <function>
            ///     }
            /// </param>

            var that = this, handler;

            Diag.assertParamIsObject(message, 'message');

            if (this._options.closeable) {
                this._element.addClass('closeable');
            }
            else {
                this._element.removeClass('closeable');
            }

            this._clear(false);
            this._messageType = message.type;
            
            this._errorHeader.html(message.header);

            if (message.content) {
                this._errorContent.html(message.content);
                this._setErrorDetailsVisibility(this._options.expanded);
                this._showErrorLink.show();
            }
            else {
                this._errorContent.hide();
                this._showErrorLink.hide();
            }

            this._element.show();

            switch (this._messageType) {
                case MessageAreaType.Info:
                    this._element.addClass('info-message');
                    break;
                case MessageAreaType.Warning:
                    this._element.addClass('warning-message');
                    break;
                case MessageAreaType.Error:
                    this._element.addClass('error-message');
                    break;
                default:
                    Diag.fail(String.format('Unexpected MessageAreaType {0}', this._messageType));
                    break;
            }

            if (message.click) {
                Diag.assertParamIsFunction(message.click, 'message.click');

                handler = function () {
                    message.click();
                    that.clear(); // as we took action we need to clear the control
                };

                this._errorHeader.addClass('clickable');

                this._errorHeader.click(handler);

                this._errorHeader.keydown(function (e) {
                    if (e.keyCode === TFS.UI.KeyCode.ENTER) {
                        handler();
                        return false;
                    }
                });
            }

            // Notify listeners that the message area has been updated.
            this._raiseDisplayComplete();
        },

        _toggle: function () {
            if (this._errorContent.is(':visible')) {
                this._setErrorDetailsVisibility(false);
            }
            else {
                this._setErrorDetailsVisibility(true);
            }
            this._raiseDisplayComplete();
        },

        _setErrorDetailsVisibility: function (show) {
            if (show) {
                this._showErrorLink.text(Resources.HideDetails);
                this._errorContent.show();
            }
            else {
                this._showErrorLink.text(Resources.ShowDetails);
                this._errorContent.hide();
            }
        },

        clear: function () {
            /// <summary>Clear the shown message</summary>
            this._clear(true);
        },

        _clear: function (raiseDisplayCompleteEvent) {
            /// <summary>Clear the shown message</summary>
            /// <param name="raiseDisplayCompleteEvent" type="String">Indicates if the display complete event should be raised.</param>
            Diag.assertParamIsBool(raiseDisplayCompleteEvent, "raiseDisplayCompleteEvent");

            // Don't need to do anything if the message has already been cleared.
            // This helps with perf in the cases when display-complete event listeners are expensive.
            if (this._errorHeader.text().length === 0) {
                return;
            }

            switch (this._messageType) {
                case MessageAreaType.Info:
                    this._element.removeClass('info-message');
                    break;
                case MessageAreaType.Warning:
                    this._element.removeClass('warning-message');
                    break;
                case MessageAreaType.Error:
                    this._element.removeClass('error-message');
                    break;
                default:
                    break;
            }
            this._messageType = MessageAreaType.None;

            this._errorHeader.text('');
            this._errorContent.text('');
            this._errorHeader.unbind('click');
            this._errorHeader.removeClass('clickable');
            this._element.hide();

            if (raiseDisplayCompleteEvent) {
                // Notify listeners that the message area has been updated.
                this._raiseDisplayComplete();
            }
        },

        _raiseDisplayComplete: function () {
            this._fire(MessageAreaControl.EVENT_DISPLAY_COMPLETE);
        }
    });

    // Begin jQuery-style widgets (originally TFS.UI.Widgets.js)

    /*
    * General functions used by the widgets to perform their roles
    */
    function doesKeyMatch(keyDefinition, event) {
        // If keyDefinition.altKey or keyDefinition.ctrlKey is undefined, ! will cast it to true
        if (keyDefinition.which === event.which &&
            (!keyDefinition.altKey === !event.altKey) &&
            (!keyDefinition.ctrlKey === !event.ctrlKey)) {
                return true;
            }
        else {
            return false;
        }
    }

    /*
    * jQuery overrides: This is a set of function that override the base
    * behavior of some of the jQuery functions.
    */
    (function ($) {

        // Override the val() function to handle watermark evaluation
        $.fn.valWatermark = $.fn.val;
        $.fn.val = function (value) {

            // If this control has the watermark class we dont want to
            // return the default value on a query of the value if it is
            // supplied by the control at the moment.
            if (this.hasClass('watermark')) {
                if (!value) {
                    return '';
                }
                else {
                    this.removeClass('watermark');
                }
            }
            return $.fn.valWatermark.apply(this, arguments);
        };
    })(jQuery);

    /*
    * The Watermark widget is used to setup a value that will be show when
    *      the input control doesn't have focus and has no current value.
    *
    * OPTIONS
    *  watermarkText: This is thest you want to appear in the input control
    *      when the input control has no value.
    */


    /*
    * The Dropdown widget allows the caller to use a highly customized dropdown
    *      combobox.
    *
    * OPTIONS
    *  dynamicSearch: false (default) shows the search icon and allows for dynamic
    *      loading of list content. false shows a drop down icon and doesnt
    *      allow dynamic loading of the list content.
    *
    *  id: The id attribute to apply to the user input element of the control.
    *      This attribute is also applied to name attribute for form submision.
    *
    *  initialValue: initial value for the input control. default is no value.
    *
    *  listSize: The maximum number of items to show before a scrollbar becomes
    *      visible.
    *
    *  searchHostKey: Optional key to initiate a search.
    *
    *  showDelete: true means the delete action should be made available for
    *      each item in the set of valid values. default is false.
    *
    *  showListOnKeystroke: true means that as text is typed that partially
    *      matches one or more filtered item the keystroke will open the list
    *      and filter the available values. false means the list must be manually
    *      opened.
    *
    *  watermarkText: Optional watermark text to place into the input control.
    *
    * EVENTS
    *  cancelSearch: is fired when the control wants the search UI to be cancelled.
    *  completeSearch: is fired when an active search has completed.
    *  startSearch: is fired when the control starts a search.
    *  querySearch: is fired when the control is requesting the parent to fill the
    *      control with the result of the dynamic query.
    *
    *  filterItem: is fired when a caller attempts to add an item to the search
    *      result. A pendingItem is passed to the callback with two properties.
    *          deny: true will prevent the item from being added, default = false.
    *          value: The value being added to the result list.
    *  removeItem: is fired when the user removes an item from the search result.
    *      The value being removed is the only property passed.
    *
    *  itemRemoved: is fired after an item has been removed from the search
    *      result. The only property passed to the event is the value being
    *      removed.
    *  valueUpdated: is fired when the controls value is updated. The only
    *      property passed to the event is the updated value.
    *  itemSelected: is fired after the valueUpdated when an item from the list
    *      is selected, the value of the item is passed to the event.
    *
    */


    /*
    *
    *  items: this is an array of elements we can will add to the list before
    *      the search is performed. This allows the control to have some static
    *      as well as some dynamic values.
    *
    */


    /// accessible plugin:
    /// use this on any element to make it keyboard accessible (with enter or space)
    /// selects next tabbable element if target element disappears
    /// :tabbable is a subset of :focusable, that filters out tabindex < 0
    (function ($) {
        $.fn.accessible = function () {
            return this.attr('tabindex', '0')
                .keyup(function (e) {
                    var tabbables, target, current;

                    if (e.keyCode === $.ui.keyCode.ENTER ||
                        e.keyCode === $.ui.keyCode.SPACE) {
                            tabbables = $(':tabbable');
                            target = $(e.target);
                            current = tabbables.index(target);
                            target.click();

                            // if target disappeared, find next tabbable
                            while (!target.is(':visible:tabbable')) {
                                target = tabbables.eq(current++);
                                if (target) {
                                    if (target.is(':visible:tabbable')) {
                                        // found target
                                        target.focus();
                                        break;
                                    }
                                    else {
                                        // target is not visible
                                        continue;
                                    }
                                }
                                else {
                                    // could not find any targets
                                    break;
                                }
                            }
                        }
                });
        };
    })(jQuery);

    // End jQuery-style widgets

    return {
        BaseComboBehavior: BaseComboBehavior,
        ComboListBehavior: ComboListBehavior,
        ComboTreeBehavior: ComboTreeBehavior,
        ComboDateBehavior: ComboDateBehavior,
        BaseComboDropPopup: BaseComboDropPopup,
        ComboListDropPopup: ComboListDropPopup,
        ComboTreeDropPopup: ComboTreeDropPopup,
        ComboDateDropPopup: ComboDateDropPopup,
        Combo: Combo,
        DatePicker: DatePicker,
        DatePanel: DatePanel,
        CollapsiblePanel: CollapsiblePanel,
        RichEditor: RichEditor,
        TreeView: TreeView,
        VerticalBreadCrumb: VerticalBreadCrumb,
        AjaxPanel: AjaxPanel,
        Dialog: Dialog,
        ModalDialog: ModalDialog,
        ConfirmationDialog: ConfirmationDialog,
        Splitter: Splitter,
        CheckboxList: CheckboxList,
        Histogram: Histogram,
        StatusIndicator: StatusIndicator,
        FilterControl: FilterControl,
        WaitControl: WaitControl,
        LongRunningOperation: LongRunningOperation,
        Clipboard: Clipboard,
        CopyContentDialog: CopyContentDialog,
        MessageAreaControl: MessageAreaControl,
        MessageAreaType: MessageAreaType,
        TileZone: TileZone
    };
});

// SIG // Begin signature block
// SIG // MIIatQYJKoZIhvcNAQcCoIIapjCCGqICAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFFmqpwolWxZi
// SIG // wby219a2Jlhj9PgxoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSoMIIEpAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIHKMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTk4P5fVnHIKWQhWCWSEKfV
// SIG // mKxQEDBqBgorBgEEAYI3AgEMMVwwWqBAgD4AVABGAFMA
// SIG // LgBVAEkALgBDAG8AbgB0AHIAbwBsAHMALgBDAG8AbQBt
// SIG // AG8AbgAuAGQAZQBiAHUAZwAuAGoAc6EWgBRodHRwOi8v
// SIG // bWljcm9zb2Z0LmNvbTANBgkqhkiG9w0BAQEFAASCAQA5
// SIG // 0JdY8dwxeiotioqhkyefWAHUxxXpJrQHoRyKjvCeknqk
// SIG // puNx4skjkXNPYH9G792jvja5Bx0CXnbW9MXZ4PqtqQyS
// SIG // oTOCPnw4rQAUTgbEDRZNCw+T34wK8DUE7wJw39TBReQZ
// SIG // 7sIixWVsDy1FT4I4lLPPny10UOPtUF1fhRssPZsUEnwG
// SIG // raeWzNPtmAHQyTx7I47Tb7JqOrNb8kxeFnTu721ksVLn
// SIG // QW++O3XQi9zZdTAmk6jF44v6JSlSVm5l5Eky3pTtj3u8
// SIG // ff84/YEKQvGS2luRTWgn4EnU25DMcKjh6bvglo0Oe04z
// SIG // 264ibG+lf8lNYW8HO7PySHDVdDwTkRpDoYICHzCCAhsG
// SIG // CSqGSIb3DQEJBjGCAgwwggIIAgEBMIGFMHcxCzAJBgNV
// SIG // BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYD
// SIG // VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
// SIG // Q29ycG9yYXRpb24xITAfBgNVBAMTGE1pY3Jvc29mdCBU
// SIG // aW1lLVN0YW1wIFBDQQIKYQKSSgAAAAAAIDAJBgUrDgMC
// SIG // GgUAoF0wGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAc
// SIG // BgkqhkiG9w0BCQUxDxcNMTMwMzE1MDYzMzU1WjAjBgkq
// SIG // hkiG9w0BCQQxFgQU2tquz4VcrQcK1deRdSRhUc57gV8w
// SIG // DQYJKoZIhvcNAQEFBQAEggEASp+Xp1zmMokePjvInNCK
// SIG // TeUIkDB5Mi9Rg7GwNWeNJrlwPO87dLJhpQl+PevydNok
// SIG // b0Rb3AwejlJucRVPCy05ghPtJoyYBOwec7A86BbBOYPf
// SIG // cl9aDbZszQMrez8ksbV8gILKZBf6CA7orkiURoBdKFZ3
// SIG // P+x/TyuCZ52QvWLgZDx3JLt4yy1rcTjdiE/yHQZHRhNA
// SIG // J8JsnpY3RHy6uGnFEC+/LjrWvYA6bWqw0EQd+osOY6A2
// SIG // RC12qZYeej9O4HzAPH8r4tQCr3UMACr1fRr2tpDb7ZzL
// SIG // v06xg2gamYnMh9E0gDb1h2vT4S0bUxP+ssdFUVLkOb7S
// SIG // UeecFm4+QotluA==
// SIG // End signature block
